/***************************************************************************
 *   Copyright (C) 2004, 2005, 2006 by Stephen McInerney                   *
 *   spm@stedee.id.au                                                      *
 *                                                                         *
 *   $Id: db_dnshistory.c 58 2006-01-02 10:40:49Z steve $
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/***************************************************************************
 ***************************************************************************
 * db_dnshistory.c
 *
 * Create, Add and Delete actions for the Visitor Database
 ***************************************************************************
 ***************************************************************************/

/***************************************************************************
 ***************************************************************************
 * ModificationHistory:
 **********************
 * 17-Mar-2005 steve     Initial Creation - taken from visitors
 *
 ***************************************************************************
 ***************************************************************************/


#include "dnshistory.h"

/************************************************************************
 * open_dnshistory_db                                                           *
 *                                                                      *
 * Creates and Opens the Berkeley DB                                    *
 * Also creates a cache of a given size (in Mb), or none if <= 0        *
 ************************************************************************
 *  db_ptr      :- double pointer to the DB structure                   *
 *  cache_size  :- Size of cache in Mb                                  *
 *                      if <= 0 will not be created                     *
 ************************************************************************/
void
open_dnshistory_db(DB ** db_ptr,                /* The database to open */
                   char *db_dirfilename,        /* Filename of the database file on disk */
                   u_int32_t cache_size,        /* the size of the cache to create for this DB - in Mb */
                   u_int32_t flags              /* DB Open Flags */
    )
{
    int rtn_db;                                 /* Return value from the various opens/creates */

    VPRINT(VERBOSE2, "%s\n", msg_v1_hashdb);
    VPRINT(VERBOSE3, "%s%s\n", msg_v2_create_hashdb, db_dirfilename);

    /* Create the DB */
    rtn_db = db_create(db_ptr, NULL, 0);
    if (rtn_db != 0) {
        ERRVPRINT(VERBOSE0, msg_F_db_create, db_strerror(rtn_db));
        exit(V_EXIT_DB_CREATION);
    }

    /* Set a bigger cache. _Significantly_ speeds things up! */
    if (cache_size > 0) {
        cache_size *= 1024 * 1024;              /* Convert to Mb */
        VPRINT(VERBOSE3, "%s%d\n", msg_v2_create_dbcache, cache_size);
        rtn_db = (*db_ptr)->set_cachesize(*db_ptr, 0, cache_size, 1);
        if (rtn_db != 0) {
            ERRVPRINT(VERBOSE0, msg_F_db_cache, db_strerror(rtn_db));
            exit(V_EXIT_DB_CACHE_CREATION);
        }
    } else {
        VPRINT(VERBOSE3, "%s%d\n", msg_v2_no_dbcache, cache_size);
    }


    VPRINT(VERBOSE3, "Open the DB: %s\n", db_dirfilename);
    /* Open the DB itself */
    /* BTrees appear to be about 11% faster - based on: 119 Million log lines, 660 000 DB Adds */
    /* rtn_db = (*db_ptr)->open (*db_ptr, NULL, db_dirfilename, NULL, DB_HASH, DB_CREATE, 0664); */
    rtn_db = (*db_ptr)->open(*db_ptr, NULL, db_dirfilename, NULL, DB_BTREE, flags, DB_PERMISSIONS);
    if (rtn_db != 0) {
        (*db_ptr)->err(*db_ptr, rtn_db, "%s", db_dirfilename);
        exit(V_EXIT_DB_OPEN);
    }
}


/************************************************************************
 * close_dnshistory_db                                                  *
 *                                                                      *
 * Closes the Berkeley DB                                               *
 ************************************************************************/
int
close_dnshistory_db(DB ** db_ptr                /* The database to open */
    )
{
    int rtn_db_close;

    rtn_db_close = (*db_ptr)->close(*db_ptr, 0);
    if (rtn_db_close != 0) {
        ERRVPRINT(VERBOSE0, msg_F_db_close, rtn_db_close);
    }
    return (rtn_db_close);
}

/************************************************************************
 * add_record                                                             *
 *                                                                      *
 * Add a Key/Data pair to the cookie hash                               *
 * This version uses thread mutexes to control adding                   *
 ************************************************************************
 *  key_val     :- Pointer to key to store                              *
 *  size_key    :- Size of this key                                     *
 *  data_val    :- Pointer to data to store, for which key is the index *
 *  size_data   :- Size of the data                                     *
 ************************************************************************/
void
add_record(DB ** db_ptr,                        /* FIXME */
           void *key_val,                       /* The key to store as a string */
           int size_key,                        /* size of that key */
           void *data_val,                      /* data "bag" */
           int size_data                        /* size of the data bag */
    )
{
    int rtn;                                    /* Return value */
    DBT dbt_key, dbt_data;                      /* DBT's for Key and Data */

    VPRINT(VERBOSE4, "  ADD Record%s\n", "");

    /* Zero DBT's in line with BDB recommendations */
    memset(&dbt_key, 0, sizeof(dbt_key));
    memset(&dbt_data, 0, sizeof(dbt_data));

    /* Set up DBT's for storing */
    dbt_key.data = (char *) key_val;
    dbt_key.size = size_key;
    dbt_data.data = (char *) data_val;
    dbt_data.size = size_data;

    /* Store the key/data and verify */
    pthread_mutex_lock(&mutex_db_access);       /* Lock DB accesses */
    rtn = (*db_ptr)->put(*db_ptr, NULL, &dbt_key, &dbt_data, 0);
    if (rtn != 0) {
        (*db_ptr)->err(*db_ptr, rtn, "\tDB->put");
        close_dnshistory_db(db_ptr);
        /* XFREE (db_ptr); */
        exit(V_EXIT_DB_PUT);
    }
    pthread_mutex_unlock(&mutex_db_access);
}


/************************************************************************
 * delete_record                                                             *
 *                                                                      *
 * Delete a Key/Data pair from the hash                                 *
 ************************************************************************
 *  key_val     :- Pointer to key to remove                             *
 ************************************************************************/
void
delete_record(DB ** db_ptr,                     /* Database Pointer */
              char *key_val                     /* Key to remove */
    )
{
    int rtn;                                    /* Return value */
    DBT del_key;

    VPRINT(VERBOSE4, "  DELETE Record%s\n", "");

    /* Zero DBT's in line with BDB recommendations */
    memset(&del_key, 0, sizeof(del_key));

    del_key.data = key_val;
    del_key.size = (strlen(key_val) + 1) * sizeof(char);

    rtn = (*db_ptr)->del(*db_ptr, NULL, &del_key, 0);
    if (rtn != 0) {
        (*db_ptr)->err(*db_ptr, rtn, "\tDB->del");
        close_dnshistory_db(db_ptr);
        XFREE(db_ptr);
        exit(V_EXIT_DB_DELETE);
    }
}


/************************************************************************
 ************************************************************************
 *                      END OF FILE                                     *
 ************************************************************************
 ************************************************************************/
