/*
 * Copyright (C) 2013, 2014
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.3
import QtQuick.XmlListModel 2.0
import Ubuntu.Components 1.1
import Ubuntu.Components.ListItems 1.0 as ListItem
import Ubuntu.Components.Popups 1.0

import "../databasemodule_v2.js" as DB
import "../."

Page {
    id: appendFeedPage

    objectName: "appendfeedpage"
    title: i18n.tr("Add feeds")
    flickable: null
    visible: false

    property bool isDirty: false // Need to clear all page content.
    property int selectedCount: 0
    property bool resultsReceived: false // Indicates that at least once results were received.

    function reloadPageContent() {
        // appendFeedPage.tools = appendFeedTools
        if (isDirty) {
            tfFeedUrl.text = ""
            resultsReceived = false
            clearModelDependData()
        }
        isDirty = false
    }

    /* Clear model and model's depend data.
     * Currently only selectedCount.
     */
    function clearModelDependData() {
        searchResultsModel.clear()
        selectedCount = 0
    }

    // -------------------------------------       GOOGLE API

    Connections {
        target: googleFeedApi

        onFindResult: {
            resultsReceived = true

            if ( result.responseStatus !== 200 ) {
                // No need to show dialog in case of abort.
                if (result.responseStatus !== 0) {
                    PopupUtils.open(errorDialogComponent, appendFeedPage,
                                    {"text" : i18n.tr("Failed to perform a feed search by keyword"),
                                        "title" : i18n.tr("Search failed")})
                    console.log(JSON.stringify(result))
                }
                return
            }

            clearModelDependData()

            var entries = result.responseData.entries

            for (var i = 0; i < entries.length; i++) {
                searchResultsModel.append({"url" : entries[i].url,
                                              "title" : entries[i].title,
                                              "description" : entries[i].contentSnippet,
                                              "link" : entries[i].link,
                                              "isSelected" : false})
            }
        }

        onLoadResult: {
            resultsReceived = true

            if ( result.responseStatus !== 200 ) {
                // No need to show dialog in case of abort.
                if (result.responseStatus !== 0) {
                    PopupUtils.open(errorDialogComponent, appendFeedPage,
                                    {"text" : i18n.tr("Failed to perform a feed search by URL"),
                                        "title" : i18n.tr("Search failed")})
                    console.log(JSON.stringify(result))
                }
                return
            }

            clearModelDependData()

            var feed = result.responseData.feed

            searchResultsModel.append({"url" : feed.feedUrl,
                                          "title" : feed.title,
                                          "description" : feed.description,
                                          "link" : feed.link,
                                          "isSelected" : false})
        }
    }

    GoogleFeedApi {
        id: googleFeedApi
    }

    // -------------------------------------       GOOGLE API

    Column {
        id: appendFeedColumn

        anchors.top: parent.top
        anchors.topMargin: units.gu(2)
        width: parent.width
        spacing: units.gu(2)

        TextField {
            objectName: "tfFeedUrl"
            id: tfFeedUrl

            placeholderText: i18n.tr("Type a keyword or URL")

            width: parent.width - units.gu(4)
            // height:units.gu(5)
            anchors {
                horizontalCenter: parent.horizontalCenter
            }
            primaryItem: Image {
                height: parent.height*0.5
                width: height
                anchors.verticalCenter: parent.verticalCenter
                //                anchors.verticalCenterOffset: -units.gu(0.2)
                source: Qt.resolvedUrl("../icons_tmp/find.svg")
                smooth: true

                MouseArea {
                    anchors.fill: parent
                    onClicked: {
                        if (Qt.inputMethod.visible)
                            tfFeedUrl.accapt()
                    }
                }
            }

            onAccepted: {
                accapt()
            }

            function accapt() {
                Qt.inputMethod.hide()
                var userInput = text

                if (userInput == "")
                    return

                // Very simple logic, URL if there are no spaces and contains dots.
                // But simple not means that it is wrong.
                var isUrlEntered = (userInput.indexOf(" ") === -1 && userInput.indexOf(".") !== -1)

                if (isUrlEntered) {
                    if (userInput.indexOf("http://") !== 0)
                        userInput = "http://" + userInput
                    googleFeedApi.loadFeed(userInput)
                }
                else googleFeedApi.findFeeds(text)
            }
        }

        ListItem.Header {

            ListItem.ThinDivider { }

            text: i18n.tr("Search results")
        }
    } // Column


    ListView {
        id: searchResults
        objectName: "feedpagelistview"

        width: parent.width
        clip: true
        anchors {
            bottom: fakePanel.top
            // bottomMargin: appendFeedTools.height
            top: appendFeedColumn.bottom
        }

        model: searchResultsModel

        delegate: ListItem.Standard {
            text: model.title
            objectName: "feedpagelistitem-" + model.title

            control: CheckBox {
                anchors.verticalCenter: parent.verticalCenter

                objectName: "feedCheckbox-" + model.title
                onCheckedChanged: {
                    searchResultsModel.setProperty(index, "isSelected", checked)
                    if (checked)
                        selectedCount++
                    else selectedCount--
                }
            }
        }
    }

    Rectangle {
        id: fakePanel

        color: "#fafafa"
        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }
        height: units.gu(6)
        Item {
            anchors.fill: parent

            Button {
                anchors {
                    verticalCenter: parent.verticalCenter
                    left: parent.left
                    leftMargin: units.gu(1)
                }
                gradient: UbuntuColors.greyGradient
                action: Action {
                    text:  i18n.tr("Cancel")

                    onTriggered: {
                        googleFeedApi.abort()
                        pageStack.pop()
                    }
                }
            }

            ActivityIndicator {
                id: checkRunning

                visible: googleFeedApi.inProgress
                running: googleFeedApi.inProgress

                anchors {
                    verticalCenter: parent.verticalCenter
                    right: nextBtn.left
                    rightMargin: units.gu(1)
                }
            }

            Button {
                id: nextBtn
                objectName: "nextButton"
                anchors {
                    verticalCenter: parent.verticalCenter
                    right: parent.right
                    rightMargin: units.gu(1)
                }

                enabled: !googleFeedApi.inProgress && selectedCount > 0
                action: Action {
                    text:  i18n.tr("Next")

                    onTriggered: {
                        if (!nextBtn.enabled)
                            return

                        var selectedFeeds = []
                        for (var i = 0; i < searchResultsModel.count; i++) {
                            var curItem = searchResultsModel.get(i)

                            if (curItem.isSelected) {
                                selectedFeeds.push(curItem)
                            }
                        }

                        pageStack.push(chooseTopicPage, {"feedsToAdd" : selectedFeeds})
                    }
                }
            } // Button
        }
    } // Rectangle fakePanel

    ListModel {
        id: searchResultsModel
    }

    Label {
        visible: searchResultsModel.count == 0 && resultsReceived
        anchors.centerIn: parent
        text: i18n.tr("No feeds")
        fontSize: "large"
    }
}
