import QtQuick 2.3
import Ubuntu.Components 1.1
import Ubuntu.Components.ListItems 1.0 as ListItem

import "../databasemodule_v2.js" as DB

ListItem.Empty {
    id: topicComponent

    height: columnContent.height
//    removable: true
//    highlightWhenPressed : true
    showDivider : true
//    clip: true

    property bool isExpanded: false
    property string topicName
    property int topicId
    property int modelIndex
    property bool isSelected: false

    readonly property alias isEditing: rowTopicContent.isEditing

    signal edit()
    signal editCanceled()
    signal feedDeleted()

//    onClicked: isExpanded = !isExpanded ;

    onStateChanged:{
        if (isExpanded){
            reloadFeed()
        }
    }

    Behavior on height {
        NumberAnimation { duration: 150; easing.type: Easing.OutQuad }
    }

    function reloadFeed (){
        feedModel.clear()
        var feeds_tag =  DB.loadFeedsFromTag(topicId)
        for (var i=0; i< feeds_tag.rows.length; i++){
            feedModel.append(feeds_tag.rows[i]) ;
        }
    }

    function cancelEdit() {
        rowTopicContent.isEditing = false
        inputTopicName.text = topicName
    }

    function confirmEdit() {
        var topicNameLocal = inputTopicName.text.replace(/^\s+|\s+$/g, '')

        if (topicNameLocal == "") {
            console.log("Empty topic name, edit aborted.")
            inputTopicName.text = labelTopicName.text
        } else {
            /* Make first letter capital.
             */
            topicNameLocal = topicNameLocal.charAt(0).toUpperCase() + topicNameLocal.slice(1)

            var result = DB.updateTag(topicComponent.topicId, topicNameLocal)
            if (result.rowsAffected === 1) {
                mainView.changeTopicName(topicComponent.topicId, topicNameLocal)
                labelTopicName.text = topicNameLocal
                topicName = topicNameLocal
            }
            else {
                inputTopicName.text = labelTopicName.text
            }
        }

        cancelEdit()
    }

    Rectangle {
        id: bg
        anchors.fill: parent
        color: "white"
        opacity: isSelected ? 0.4 : 0
    }

    Column{
        id: columnContent
        anchors{ left: parent.left; right: parent.right }

        /*
          topic item
        */
        Item
        {
            anchors{ left: parent.left; right: parent.right }
            height: units.gu(5)

            /*
              topic header for showing name and swipe remove
            */
            ListItem.Base{
                id: headerTopic
                objectName: "headerTopic"
                enabled: !rowTopicContent.isEditing
                removable: true
                confirmRemoval: true
                height: units.gu(5)

                onClicked: {
                    isExpanded = !isExpanded
                    editCanceled()
                }

                onItemRemoved:{
                    console.log("item about to be removed: ", topicComponent.topicId)
                    //                    itemCancelDelete.startCounting()
                    var result = DB.deleteFeedByTagId(topicComponent.topicId)
                    if (result.rowsAffected == undefined)
                    {
                        topicManagement.reloadTopics()
                    }
                    result = DB.deleteTag(topicComponent.topicId)
                    //                            topicManagement.reloadTopics()
                    if (result.rowsAffected == 1)
                    {
                        topicManagement.removeModelItem(topicComponent.modelIndex)
                    }
                    else
                    {
                        topicManagement.reloadTopics()
                    }
                }
            }

            /*
              enable if edit mode active
            */
            Row{
                id: rowTopicContent
                anchors
                {
                    top: parent.top; bottom: parent.bottom; left: parent.left;
                    leftMargin: units.gu(1); topMargin: units.gu(0.7); bottomMargin: units.gu(1);
                }
                spacing: units.gu(2)

                property bool isEditing: false

                Image{
                    anchors
                    {
                        top: parent.top; bottom: parent.bottom;
                        topMargin: units.gu(0.5); bottomMargin: units.gu(0.5);
                    }
                    fillMode: Image.PreserveAspectFit
                    source: Qt.resolvedUrl("../icons_tmp/compose.svg")
                    smooth: true

                    MouseArea{
                        enabled: !rowTopicContent.isEditing
                        anchors { fill: parent; margins: units.gu(-1) }
                        onClicked:
                        {
                            rowTopicContent.isEditing = true
                            topicComponent.edit()
                            inputTopicName.focus = true
                        }
                    }
                }

                Label{
                    id: labelTopicName
                    objectName: "labelTopicName"
                    anchors.verticalCenter: parent.verticalCenter
                    text: topicName
                    width: rowTopicContent.isEditing ? 0 : paintedWidth
                    opacity: rowTopicContent.isEditing ? 0 : 1

                    //                    Behavior on width { UbuntuNumberAnimation{} }
                }

                TextField{
                    id: inputTopicName
                    anchors.verticalCenter: parent.verticalCenter
                    text: topicName
                    width: rowTopicContent.isEditing ? headerTopic.width - units.gu(11) : 0
                    opacity: rowTopicContent.isEditing ? 1 : 0
                    hasClearButton: true
                    activeFocusOnPress: true

                    Behavior on width { UbuntuNumberAnimation{} }

                    onAccepted: {
                        console.log("accepted?")
                        Qt.inputMethod.hide()
                    }

                    Keys.enabled: rowTopicContent.isEditing
                    Keys.onPressed: {
                        event.accepted = false
                        if (event.key == Qt.Key_Return) {
                            Qt.inputMethod.hide()
                        }
                    }
                }
            }

            Image{
                id: imgArrow
                anchors
                {
                    right: parent.right; top: parent.top; bottom: parent.bottom;
                    topMargin: units.gu(1.5); bottomMargin: units.gu(1.5); rightMargin: units.gu(2)
                }
                fillMode: Image.PreserveAspectFit
                source: Qt.resolvedUrl("../icons_tmp/go-to.svg")
                smooth: true

                Behavior on rotation { UbuntuNumberAnimation{} }

                states: [
                    State {
                        name: "expanded"
                        when: topicComponent.isExpanded

                        PropertyChanges
                        {
                            target: imgArrow
                            rotation: 90
                        }
                    }
                ]
            }


        }

        /*
          feeds listview
        */
        ListView
        {
            id: feedList
            anchors{ left: parent.left; right: parent.right }
            height: 0
            opacity: 0
            interactive: false
            clip: true

            property int transitionDuration: 250

            Behavior on opacity {
                PropertyAnimation { duration: 150; easing.type: Easing.OutQuad }
            }

            model: VisualDataModel {
                id: visualModel
                model: ListModel {
                    id: feedModel
                }
                delegate: MouseArea {
                    id: delegateFeed

//                    anchors{ left: parent.left; right: parent.right }
                    width: feedList.width
                    height: feedItem.height

                    property int visualIndex: VisualDataModel.itemsIndex
                    property int feedId: model.id
                    property int topicId: topicComponent.topicId

                    property int startX: feedItem.x
                    property int startY: feedItem.y
                    property int startMouseY: -1

                    property real itemx
                    property real itemy
                    property var topicItem

                    // Allow dragging on the X axis for swipeDelete if not reordering
                    drag.target: feedItem
                    drag.axis: topicList.flickState == "reorder" ? Drag.YAxis : Drag.XAxis
                    drag.minimumX: topicList.flickState == "reorder" ? 0 : -feedItem.width
                    drag.maximumX: topicList.flickState == "reorder" ? 0 : feedItem.width

                    onMouseXChanged: {
                        // Only allow XChange if not in reorder state
                        if (topicList.flickState == "reorder")
                        {
                            return;
                        }

                        // New X is less than start so swiping left
                        if (feedItem.x < startX)
                        {
                            swipeBackground.state = "swipingLeft";
                        }
                        // New X is greater sow swiping right
                        else if (feedItem.x > startX)
                        {
                            swipeBackground.state = "swipingRight";
                        }
                        // Same so reset state back to normal
                        else
                        {
                            swipeBackground.state = "normal";
                            topicList.flickState = "normal";
                        }
                    }

                    onMouseYChanged: {
                        if (topicList.flickState == "normal")
                        {
                            return;
                        }

                        itemx = feedItem.width / 2
                        itemy = feedItem.y + feedItem.height / 2
                        topicItem = topicList.itemAt(itemx, itemy)
                        if (topicItem)
                            topicList.feedSelectedIndex = topicList.indexAt(itemx, itemy)
                        topicList.scrollView(itemy)
                    }

                    onPressed: {
                        startX = feedItem.x;
                        startY = feedItem.y;
                        startMouseY = mouse.y;
                    }

                    onPressAndHold: {
                        topicList.flickState = "reorder";  // enable reordering state
                        topicList.collapseAll()
                        feedItem.opacity = 0.5
                    }

                    onClicked:
                    {
                        mainView.editFeed(model.id, model.title, model.source, delegateFeed.topicId) ;
                    }

                    onReleased: {
                        topicList.feedSelectedIndex = -1
                        // Get current state to determine what to do
                        if (topicList.flickState == "reorder")
                        {
                            //feedItem.Drag.drop()  // useless until Drag.active bug fixed
                            if (topicItem)
                                topicItem.feedDrop(feedId, topicId)
                            else
                                reloadFeed()
                        }
                        else if (swipeBackground.state == "swipingLeft" || swipeBackground.state == "swipingRight")
                        {
                            // Remove if moved > 10 units otherwise reset
                            if (Math.abs(feedItem.x - startX) > units.gu(10))
                            {
                                /*
                                 * Remove the listitem
                                 *
                                 * Remove the listitem to relevant side (playlistTracksRemoveAnimation)
                                 * Reduce height of listitem and remove the item
                                 *   (swipeDeleteAnimation [called on playlistTracksRemoveAnimation complete])
                                 */
                                swipeBackground.runSwipeDeletePrepareAnimation();  // fade out the clear text
                                playlistTracksRemoveAnimation.start();  // remove item from listview
                            }
                            else
                            {
                                /*
                                 * Reset the listitem
                                 *
                                 * Remove the swipeDelete to relevant side (swipeResetAnimation)
                                 * Reset the listitem to the centre (playlistTracksResetAnimation)
                                 */
                                playlistTracksResetAnimation.start();  // reset item position
                            }
                        }

                        // ensure states are normal
                        swipeBackground.state = "normal";
                        topicList.flickState = "normal";
                    }

                    FeedComponent
                    {
                        id: feedItem
                        text: model.title
                        feedId: model.id
                        topicId: delegateFeed.topicId
                        width: topicList.width
                        height: units.gu(6)
//                        removable: true

//                        Drag.active: delegateFeed.drag.active      //  serious bug, will crash in device
                        Drag.source: delegateFeed

                        /*
                          thanks for music app provide this custom component
                          See http://pad.lv/1311800
                        */
                        SwipeDelete {
                            id: swipeBackground
                            duration: feedList.transitionDuration

                            onDeleteStateChanged: {
                                if (deleteState === true)
                                {
                                    var result = DB.deleteFeed(feedId)
                                    if (result.rowsAffected == 1)
                                        feedDeleted()
                                    topicComponent.reloadFeed()
                                }
                            }
                        }

                        states: [
                            State {
                                name: "draging"
                                when: topicList.flickState == "reorder"
                                ParentChange {
                                    target: feedItem
                                    parent: topicList
                                }

                                PropertyChanges {
                                    target: feedItem
                                    opacity: 0
                                }
                            },

                            State
                            {
                                name: ""
                                ParentChange {
                                    target: feedItem
                                    parent: delegateFeed
                                }

                                PropertyChanges {
                                    target: feedItem
                                    opacity: 1
                                }
                            }

                        ]
                    }

                    // Animation to reset the x, y of the item
                    ParallelAnimation {
                        id: playlistTracksResetAnimation
                        running: false
                        NumberAnimation {  // reset X
                            target: feedItem
                            property: "x"
                            to: delegateFeed.startX
                            duration: feedList.transitionDuration
                        }
                        NumberAnimation {  // reset Y
                            target: feedItem
                            property: "y"
                            to: delegateFeed.startY
                            duration: feedList.transitionDuration
                        }
                    }

                    /*
                     * Animation to remove an item from the list
                     * - Removes listitem to relevant side
                     * - Calls swipeDeleteAnimation to delete the listitem
                     */
                    NumberAnimation {
                        id: playlistTracksRemoveAnimation
                        target: feedItem
                        property: "x"
                        to: swipeBackground.state == "swipingRight" ? feedItem.width : 0 - feedItem.width
                        duration: feedList.transitionDuration

                        onRunningChanged: {
                            // Remove from queue once animation has finished
                            if (running == false)
                            {
                                swipeBackground.runSwipeDeleteAnimation();
                            }
                        }
                    }


                }


                // feedlistDelegate

            }
        }
    }



    states: [
        State {
            name: "expanded"
            when: isExpanded
            PropertyChanges
            {
                target: feedList
                height: feedList.contentItem.childrenRect.height
                opacity: 1
            }
        },

        State {
            name: ""
//            when: isExpanded
            PropertyChanges
            {
                target: feedList
                height: 0
                opacity: 0
            }
        }
    ]
}
