import QtQuick 2.3
import Ubuntu.Components 1.1
import Ubuntu.Components.ListItems 1.0 as ListItem
import Ubuntu.Components.Popups 1.0
import Ubuntu.Layouts 1.0

import "../databasemodule_v2.js" as DB

Page {
    id: topicManagement
    objectName: "topicmanagement"

    title: i18n.tr("Edit topics")
    flickable: content
    tools: toolbar

    signal feedEdit(int topicId)
    signal topicDeleted()

    property int editIndex

    Component.onCompleted: {
        reloadTopics ()
    }

    function reloadTopics ()
    {
        topicModel.clear()
        var topics = DB.loadTags()
        for (var i=0; i < topics.rows.length; i++){
            topicModel.append(
                    {
                "id": topics.rows[i].id
                ,"name": topics.rows[i].name
            })
        }
    }

    function removeModelItem (modelIndex){
        topicModel.remove(modelIndex)
        topicDeleted()
    }

    function reloadPageContent() {
        topicManagement.flickable = null
        topicManagement.flickable = content
    }

    Rectangle {
        id: fakePanel

        color: "#fafafa"
        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }
        height: units.gu(6)
        Item {
            anchors.fill: parent

            Button {
                id: cancelButton
                text: i18n.tr("Cancel")
                gradient: UbuntuColors.greyGradient
                anchors {
                    verticalCenter: parent.verticalCenter
                    left: parent.left
                    leftMargin: units.gu(1)
                }
                visible: topicManagement.state == "editMode"
                onClicked: {
                    Qt.inputMethod.hide()
                    topicList.currentItem.cancelEdit()
                    topicManagement.state = "default"
                    // editConfirm.visible = false
                    topicList.currentIndex = -1
                    timerHeader.start()
                }
            }

            Button {
                id: confirmButton
                text: i18n.tr("Confirm")
                anchors {
                    verticalCenter: parent.verticalCenter
                    right: parent.right
                    rightMargin: units.gu(1)
                }
                visible: topicManagement.state == "editMode"
                onClicked: {
                    topicList.currentItem.confirmEdit()
                    topicManagement.state = "default"
                    // editConfirm.visible = false
                    topicList.currentIndex = -1
                }
            }

            Button {
                id: actionsButton
                text: i18n.tr("Add Feed")
                anchors {
                    verticalCenter: parent.verticalCenter
                    right: parent.right
                    rightMargin: units.gu(1)
                }
                visible: topicManagement.state != "editMode"
                onClicked: pageStack.push(appendFeedPage/*, {"isDirty" : true}*/)
            }
        }
    } // Rectangle fakePanel

    /*
      main content
    */
    Flickable {
        id: content
        anchors {
            left: parent.left
            right: parent.right
            top: parent.top
            bottom: fakePanel.top
        }
        contentHeight: contentItem.childrenRect.height
//        boundsBehavior: (contentHeight > topicManagement.height) ? Flickable.DragAndOvershootBounds : Flickable.StopAtBounds

        /*
          idea from music app, set interactive to false may work
        */
        state: "normal"
        states: [
            State {
                name: "normal"
                PropertyChanges {
                    target: content
                    interactive: true
                }
            },
            State {
                name: "reorder"
                PropertyChanges {
                    target: content
                    interactive: false
                }
            }
        ]

        Column{
            anchors{ left: parent.left; right: parent.right }

            /*
              add topic button
            */
            Item {
                id: itemAddTopic
                objectName: "addTopic"
                anchors{ left: parent.left; right: parent.right }
                height: units.gu(9)

                Button {
                    id: btnAddTopic

                    objectName: "btnAddTopic"
                    text: i18n.tr("Add topic")
                    anchors { fill: parent; margins: units.gu(2) }
                    onClicked: pageStack.push(createTopicPage)
                }
            }

            /*
              topic listview
            */
            ListView {
                id: topicList
                anchors{ left: parent.left; right: parent.right }
                height: contentItem.childrenRect.height
                interactive: false
                currentIndex: -1
//                clip: true

                property int feedSelectedIndex: -1
                property alias flickState: content.state
                property real delegateRootY

                signal collapseAllItem()

                function collapseAll(){
                    collapseAllItem()
                }

                function scrollView(itemy) {
//                    console.log("itemy: ", itemy - content.contentY)
                    if ((itemy - content.contentY) < units.gu(16)) {
                        if (!content.atYBeginning)
                            content.contentY -= 3
                    }
                    else if ((itemy - content.contentY) > ( content.height - units.gu(16))) {
                        if (!content.atYEnd)
                            content.contentY += 3
                    }
                }

                displaced: Transition {
                    NumberAnimation { properties: "x,y"; easing.type: Easing.OutQuad }
                }

                model: VisualDataModel {
                    id: visualModel
                    model: ListModel {
                        id: topicModel
                    }
                    delegate: MouseArea {
                        id: delegateRoot

                        property int visualIndex: VisualDataModel.itemsIndex
                        property alias isExpanded: topicItem.isExpanded
                        property alias isEditing: topicItem.isEditing
                        property bool isSelected: visualIndex == topicList.feedSelectedIndex

                        anchors{ left: parent.left; right: parent.right }
                        height: topicItem.height

                        function cancelEdit() {
                            topicItem.cancelEdit()
                        }

                        function confirmEdit() {
                            topicItem.confirmEdit()
                        }

                        function feedDrop(feedId, topicId) {
                            console.log("dropped item feed id", feedId)
                            console.log("dropped item topic id", topicId)
                            DB.deleteFeedTag(feedId, topicId)
                            DB.addFeedTag(feedId, model.id)
                            delegateRoot.isExpanded = true
                            feedEdit(topicId)
                            feedEdit(model.id)
                        }

                        TopicComponent{
                            id: topicItem
                            topicName: model.name
                            topicId: model.id
                            modelIndex: index
//                            isExpanded: delegateRoot.isExpanded

                            onEdit: {
                                if (topicList.currentItem)
                                    topicList.currentItem.cancelEdit()
                                topicManagement.editIndex = index
                                topicManagement.state = "editMode"
                                // editConfirm.visible = true
                                topicList.currentIndex = index
                            }

                            onEditCanceled: {
                                if (topicList.currentItem)
                                    topicList.currentItem.cancelEdit()
                                topicManagement.state = "default"
                                // editConfirm.visible = false
                                topicList.currentIndex = -1
                            }

                            onFeedDeleted: {
                                feedEdit(model.id)
                            }

                            states: [
                                State {
                                    name: ""
                                    PropertyChanges {
                                        target: topicItem
                                        isSelected: false
                                    }
                                },
                                State {
                                    name: "selected"
                                    when: delegateRoot.isSelected
                                    PropertyChanges {
                                        target: topicItem
                                        isSelected: true
                                    }
                                }
                            ]
                        }

                        /*
                          DropArea is useless here, because Drag.active will lead crash in device
                          so I use another more complicated way to act like DropArea
                        */
                        DropArea {
                            id: dropArea
                            anchors { fill: parent}

                            onEntered:{
                                console.log("enter", drag.source)
                                topicItem.selected = true
                            }

                            onExited:{
//                                console.log("exit", drag.source)
                                topicItem.selected = false
                            }

                            onDropped:{
//                                console.log("drop", drag.source)
                                topicItem.selected = false
                                drag.source.opacity = 0
                                console.log("dropped item feed id", drag.source.feedId)
                                console.log("dropped item topic id", drag.source.topicId)
                                DB.deleteFeedTag(drag.source.feedId, drag.source.topicId)
                                DB.addFeedTag(drag.source.feedId, model.id)
                                delegateRoot.isExpanded = true
                            }
                        }

                        Connections {
                            id: connKeyBroad
                            target: Qt.inputMethod

                            onVisibleChanged: {
                                if (pageStack && (pageStack.currentPage == topicManagement) && Qt.inputMethod.visible) {
                                    topicList.delegateRootY = topicList.currentItem.y
                                    timerKeyboard.stop()
                                    timerKeyboard.start()
                                }
                            }
                        }

                        Connections{
                            id: connTopicList
                            target: topicList

                            onCollapseAllItem:
                            {
                                delegateRoot.isExpanded = false
                            }
                        }

                        Connections{
                            id: connEditFeed0
                            target: editFeed

                            onApply:{
                                if (model.id == newTopicId || model.id == previousTopicId){
                                    topicItem.reloadFeed()
                                    topicItem.isExpanded = true
                                    feedEdit(previousTopicId)
                                    feedEdit(newTopicId)
                                }
                            }

                            onDeleteFeed:{
                                if (model.id == topicId){
                                    topicItem.reloadFeed()
                                    topicItem.isExpanded = true
                                    feedEdit(topicId)
                                }
                            }
                        }

                    }
                }
            } // listview

        } // column
    } // flickable

    /* a bad workaround for forcing TextField to show on the top of keyboard
       flick the listview doesn't work because the keyboard needs about 400ms to show from bottom to top
       if better workarounds come out, pls use it to replace the current one
    */
    Timer {
        id: timerKeyboard
        interval: 400; running: false; repeat: false; triggeredOnStart: false
        onTriggered: anchorToKeyboard()

        function anchorToKeyboard() {
            if (topicList.currentItem.isEditing && Qt.inputMethod.visible) {
                var kbTemp = mainView.height - mainView.header.height - Qt.inputMethod.keyboardRectangle.height - /*editConfirm.height*/ fakePanel.height
                var yChange = topicList.delegateRootY + topicList.y - kbTemp + topicList.currentItem.height
                if (yChange > 0) {
                    content.contentY += yChange
                }
            }
        }
    }

    Timer {
        id: timerHeader
        interval: 500; running: false; repeat: false; triggeredOnStart: false
        onTriggered: topicManagement.header.show()
    }

    Connections{
        id: connAddTopic
        target: createTopicPage

        onTopicAdded:{
            reloadTopics()
        }
    }

    Connections {
        id: connAddFeed
        target: chooseTopicPage

        onTopicChoosen: {
            reloadTopics()
            for (var i=0; i<topicModel.count; i++){
                if (topicModel.get(i).id == topicId){
                    topicList.currentIndex = i
                    topicList.currentItem.isExpanded = true
                }
            }
        }
    }

    states: [
        State {
            name: "default"
//            PropertyChanges {
//                target: topicManagement
//                tools: toolbar
//            }
        },
        State {
            name: "editMode"
//            PropertyChanges {
//                target: topicManagement
//                tools: editConfirm
//            }
        }
    ]

}
