# Copyright (C) 2013-2014 Canonical Ltd.
# Author: Barry Warsaw <barry@ubuntu.com>

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""DBus service."""

__all__ = [
    'Loop',
    'Service',
    'log_and_exit',
    ]


import os
import sys
import logging

from datetime import datetime
from dbus.service import Object, method, signal
from functools import wraps
from gi.repository import GLib
from systemimage.api import Mediator
from systemimage.config import config
from systemimage.helpers import last_update_date, version_detail
from systemimage.settings import Settings
from threading import Lock


EMPTYSTRING = ''
log = logging.getLogger('systemimage')
dbus_log = logging.getLogger('systemimage.dbus')


def log_and_exit(function):
    """Decorator for D-Bus methods to handle tracebacks.

    Put this *above* the @method or @signal decorator.  It will cause
    the exception to be logged and the D-Bus service will exit.
    """
    @wraps(function)
    def wrapper(*args, **kws):
        try:
            dbus_log.info('>>> {}', function.__name__)
            retval = function(*args, **kws)
            dbus_log.info('<<< {}', function.__name__)
            return retval
        except:
            dbus_log.info('!!! {}', function.__name__)
            dbus_log.exception('Error in D-Bus method')
            self = args[0]
            assert isinstance(self, Service), args[0]
            sys.exit(1)
    return wrapper


class Loop:
    """Keep track of the main loop."""

    def __init__(self):
        self._loop = GLib.MainLoop()
        self._quitter = None

    def keepalive(self):
        if self._quitter is not None:
            GLib.source_remove(self._quitter)
            self._quitter = None
        self._quitter = GLib.timeout_add_seconds(
            config.dbus.lifetime.total_seconds(),
            self.quit)

    def quit(self):
        if self._quitter is not None:
            GLib.source_remove(self._quitter)
            self._quitter = None
        self._loop.quit()

    def run(self):
        self._loop.run()


class Service(Object):
    """Main dbus service."""

    def __init__(self, bus, object_path, loop):
        super().__init__(bus, object_path)
        self._loop = loop
        self._api = Mediator(self._progress_callback)
        log.info('Mediator created {}', self._api)
        self._checking = Lock()
        self._update = None
        self._downloading = False
        self._paused = False
        self._rebootable = False
        self._failure_count = 0
        self._last_error = ''

    @log_and_exit
    def _check_for_update(self):
        # Asynchronous method call.
        log.info('Enter _check_for_update()')
        self._update = self._api.check_for_update()
        # Do we have an update and can we auto-download it?
        downloading = False
        if self._update.is_available:
            settings = Settings()
            auto = settings.get('auto_download')
            log.info('Update available; auto-download: {}', auto)
            if auto in ('1', '2'):
                # XXX When we have access to the download service, we can
                # check if we're on the wifi (auto == '1').
                GLib.timeout_add(50, self._download)
                downloading = True
        self.UpdateAvailableStatus(
            self._update.is_available,
            downloading,
            self._update.version,
            self._update.size,
            self._update.last_update_date,
            self._update.error)
        # Stop GLib from calling this method again.
        return False

    # 2013-07-25 BAW: should we use the rather underdocumented async_callbacks
    # argument to @method?
    @log_and_exit
    @method('com.canonical.SystemImage')
    def CheckForUpdate(self):
        """Find out whether an update is available.

        This method is used to explicitly check whether an update is
        available, by communicating with the server and calculating an
        upgrade path from the current build number to a later build
        available on the server.

        This method runs asynchronously and thus does not return a result.
        Instead, an `UpdateAvailableStatus` signal is triggered when the check
        completes.  The argument to that signal is a boolean indicating
        whether the update is available or not.
        """
        self._loop.keepalive()
        # Check-and-acquire the lock.
        log.info('test and acquire checking lock')
        if not self._checking.acquire(blocking=False):
            # Check is already in progress, so there's nothing more to do.  If
            # there's status available (i.e. we are in the auto-downloading
            # phase of the last CFU), then send the status.
            if self._update is not None:
                self.UpdateAvailableStatus(
                    self._update.is_available,
                    self._downloading,
                    self._update.version,
                    self._update.size,
                    self._update.last_update_date,
                    "")
            log.info('checking lock not acquired')
            return
        log.info('checking lock acquired')
        # We've now acquired the lock.  Reset any failure or in-progress
        # state.  Get a new mediator to reset any of its state.
        self._api = Mediator(self._progress_callback)
        log.info('Mediator recreated {}', self._api)
        self._failure_count = 0
        self._last_error = ''
        # Arrange for the actual check to happen in a little while, so that
        # this method can return immediately.
        GLib.timeout_add(50, self._check_for_update)

    @log_and_exit
    def _progress_callback(self, received, total):
        # Plumb the progress through our own D-Bus API.  Our API is defined as
        # signalling a percentage and an eta.  We can calculate the percentage
        # easily, but the eta is harder.  For now, we just send 0 as the eta.
        percentage = received * 100 // total
        eta = 0
        self.UpdateProgress(percentage, eta)

    @log_and_exit
    def _download(self):
        if self._downloading and self._paused:
            self._api.resume()
            self._paused = False
            log.info('Download previously paused')
            return
        if (self._downloading                           # Already in progress.
            or self._update is None                     # Not yet checked.
            or not self._update.is_available            # No update available.
            ):
            log.info('Download already in progress or not available')
            return
        if self._failure_count > 0:
            self._failure_count += 1
            self.UpdateFailed(self._failure_count, self._last_error)
            log.info('Update failures: {}; last error: {}',
                     self._failure_count, self._last_error)
            return
        self._downloading = True
        log.info('Update is downloading')
        try:
            # Always start by sending a UpdateProgress(0, 0).  This is
            # enough to get the u/i's attention.
            self.UpdateProgress(0, 0)
            self._api.download()
        except Exception:
            log.exception('Download failed')
            self._failure_count += 1
            # Set the last error string to the exception's class name.
            exception, value = sys.exc_info()[:2]
            # if there's no meaningful value, omit it.
            value_str = str(value)
            name = exception.__name__
            self._last_error = ('{}'.format(name)
                                if len(value_str) == 0
                                else '{}: {}'.format(name, value))
            self.UpdateFailed(self._failure_count, self._last_error)
        else:
            log.info('Update downloaded')
            self.UpdateDownloaded()
            self._failure_count = 0
            self._last_error = ''
            self._rebootable = True
        self._downloading = False
        log.info('releasing checking lock from _download()')
        try:
            self._checking.release()
        except RuntimeError:
            # 2014-09-11 BAW: We don't own the lock.  There are several reasons
            # why this can happen including: 1) the client canceled the
            # download while it was in progress, and CancelUpdate() already
            # released the lock; 2) the client called DownloadUpdate() without
            # first calling CheckForUpdate(); 3) the client called DU()
            # multiple times in a row but the update was already downloaded and
            # all the file signatures have been verified.  I can't think of
            # reason why we shouldn't just ignore the double release, so
            # that's what we do.  See LP: #1365646.
            pass
        log.info('released checking lock from _download()')
        # Stop GLib from calling this method again.
        return False

    @log_and_exit
    @method('com.canonical.SystemImage')
    def DownloadUpdate(self):
        """Download the available update.

        The download may be canceled during this time.
        """
        # Arrange for the update to happen in a little while, so that this
        # method can return immediately.
        self._loop.keepalive()
        GLib.timeout_add(50, self._download)

    @log_and_exit
    @method('com.canonical.SystemImage', out_signature='s')
    def PauseDownload(self):
        """Pause a downloading update."""
        self._loop.keepalive()
        if self._downloading:
            self._api.pause()
            self._paused = True
            error_message = ''
        else:
            error_message = 'not downloading'
        return error_message

    @log_and_exit
    @method('com.canonical.SystemImage', out_signature='s')
    def CancelUpdate(self):
        """Cancel a download."""
        self._loop.keepalive()
        # During the download, this will cause an UpdateFailed signal to be
        # issued, as part of the exception handling in _download().  If we're
        # not downloading, then no signal need be sent.  There's no need to
        # send *another* signal when downloading, because we never will be
        # downloading by the time we get past this next call.
        self._api.cancel()
        # If we're holding the checking lock, release it.
        try:
            log.info('releasing checking lock from CancelUpdate()')
            self._checking.release()
            log.info('released checking lock from CancelUpdate()')
        except RuntimeError:
            # We're not holding the lock.
            pass
        # XXX 2013-08-22: If we can't cancel the current download, return the
        # reason in this string.
        return ''

    @log_and_exit
    def _apply_update(self):
        self._loop.keepalive()
        if not self._rebootable:
            command_file = os.path.join(
                config.updater.cache_partition, 'ubuntu_command')
            if not os.path.exists(command_file):
                # Not enough has been downloaded to allow for a reboot.
                self.Rebooting(False)
                return
        self._api.reboot()
        # This code may or may not run.  We're racing against the system
        # reboot procedure.
        self._rebootable = False
        self.Rebooting(True)

    @log_and_exit
    @method('com.canonical.SystemImage')
    def ApplyUpdate(self):
        """Apply the update, rebooting the device."""
        GLib.timeout_add(50, self._apply_update)
        return ''

    @log_and_exit
    @method('com.canonical.SystemImage', out_signature='isssa{ss}')
    def Info(self):
        self._loop.keepalive()
        return (config.build_number,
                config.device,
                config.channel,
                last_update_date(),
                version_detail())

    @log_and_exit
    @method('com.canonical.SystemImage', out_signature='a{ss}')
    def Information(self):
        self._loop.keepalive()
        settings = Settings()
        current_build_number = str(config.build_number)
        response = dict(
            current_build_number=current_build_number,
            device_name=config.device,
            channel_name=config.channel,
            last_update_date=last_update_date(),
            version_detail=getattr(config.service, 'version_detail', ''),
            last_check_date=settings.get('last_check_date'),
            )
        if self._update is None:
            response['target_build_number'] = '-1'
        elif not self._update.is_available:
            response['target_build_number'] = current_build_number
        else:
            response['target_build_number'] = str(self._update.version)
        return response

    @log_and_exit
    @method('com.canonical.SystemImage', in_signature='ss')
    def SetSetting(self, key, value):
        """Set a key/value setting.

        Some values are special, e.g. min_battery and auto_downloads.
        Implement these special semantics here.
        """
        self._loop.keepalive()
        if key == 'min_battery':
            try:
                as_int = int(value)
            except ValueError:
                return
            if as_int < 0 or as_int > 100:
                return
        if key == 'auto_download':
            try:
                as_int = int(value)
            except ValueError:
                return
            if as_int not in (0, 1, 2):
                return
        settings = Settings()
        old_value = settings.get(key)
        settings.set(key, value)
        if value != old_value:
            # Send the signal.
            self.SettingChanged(key, value)

    @log_and_exit
    @method('com.canonical.SystemImage', in_signature='s', out_signature='s')
    def GetSetting(self, key):
        """Get a setting."""
        self._loop.keepalive()
        return Settings().get(key)

    @log_and_exit
    @method('com.canonical.SystemImage')
    def FactoryReset(self):
        self._api.factory_reset()
        # This code may or may not run.  We're racing against the system
        # reboot procedure.
        self.Rebooting(True)

    @log_and_exit
    @method('com.canonical.SystemImage')
    def ProductionReset(self):
        self._api.production_reset()
        # This code may or may not run.  We're racing against the system
        # reboot procedure.
        self.Rebooting(True)

    @log_and_exit
    @method('com.canonical.SystemImage')
    def Exit(self):
        """Quit the daemon immediately."""
        self._loop.quit()

    @log_and_exit
    @signal('com.canonical.SystemImage', signature='bbsiss')
    def UpdateAvailableStatus(self,
                              is_available, downloading,
                              available_version, update_size,
                              last_update_date,
                              error_reason):
        """Signal sent in response to a CheckForUpdate()."""
        # For .Information()'s last_check_date value.
        iso8601_now = datetime.now().replace(microsecond=0).isoformat(sep=' ')
        Settings().set('last_check_date', iso8601_now)
        log.debug('EMIT UpdateAvailableStatus({}, {}, {}, {}, {}, {})',
                  is_available, downloading, available_version, update_size,
                  last_update_date, repr(error_reason))
        self._loop.keepalive()

    @log_and_exit
    @signal('com.canonical.SystemImage', signature='id')
    def UpdateProgress(self, percentage, eta):
        """Download progress."""
        log.debug('EMIT UpdateProgress({}, {})', percentage, eta)
        self._loop.keepalive()

    @log_and_exit
    @signal('com.canonical.SystemImage')
    def UpdateDownloaded(self):
        """The update has been successfully downloaded."""
        log.debug('EMIT UpdateDownloaded()')
        self._loop.keepalive()

    @log_and_exit
    @signal('com.canonical.SystemImage', signature='is')
    def UpdateFailed(self, consecutive_failure_count, last_reason):
        """The update failed for some reason."""
        log.debug('EMIT UpdateFailed({}, {})',
                  consecutive_failure_count, repr(last_reason))
        self._loop.keepalive()

    @log_and_exit
    @signal('com.canonical.SystemImage', signature='i')
    def UpdatePaused(self, percentage):
        """The download got paused."""
        log.debug('EMIT UpdatePaused({})', percentage)
        self._loop.keepalive()

    @log_and_exit
    @signal('com.canonical.SystemImage', signature='ss')
    def SettingChanged(self, key, new_value):
        """A setting value has change."""
        log.debug('EMIT SettingChanged({}, {})', repr(key), repr(new_value))
        self._loop.keepalive()

    @log_and_exit
    @signal('com.canonical.SystemImage', signature='b')
    def Rebooting(self, status):
        """The system is rebooting."""
        # We don't need to keep the loop alive since we're probably just going
        # to shutdown anyway.
        log.debug('EMIT Rebooting({})', status)
