/*
 * Copyright (C) 2013 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Raúl Yeguas <neokore@gmail.com>
 *              Martin Borho <martin@borho.net>
 *              Andrew Starr-Bochicchio <a.starr.b@gmail.com>
 */
import QtQuick 2.3
import Ubuntu.Components 1.1
import Ubuntu.Components.Popups 1.0
import Ubuntu.Components.ListItems 1.0 as ListItem
import "CountryCodes.js" as Countries
import "WeatherApi.js" as WeatherApi
import "CitiesList.js" as Cities

Page {
    id: addLocationPage
    objectName: "AddLocationPage"
    title: i18n.tr("Add city")
    anchors.fill: parent
    visible: false

    readonly property real headerHeight: units.gu(9.5)
    readonly property real bottomHeight: units.gu(9.5)

    property variant addCallback: null

    Connections {
        target: mainView
        onEscapeKey: {
            pageStack.pop()
            clear()
        }
    }

    head.backAction: Action {
        iconName: "back"
        visible: true
        onTriggered: {
            pageStack.pop()
        }
    }

    Component.onCompleted: {
        Cities.preList.forEach(function(loc) {
            citiesModel.append(loc)
        })
    }

    function searchResponseHandler(messageObject) {
        searching.running = false;
        if(!messageObject.error) {
            listView.visible = true
            var results = [],
                    a1Counts = {},
            a3Counts = {},
            location;
            // build a temp list and count the adminAreas
            messageObject.result.locations.forEach(function(loc) {
                a1Counts[loc.name+loc.adminName1] = (!a1Counts[loc.name+loc.adminName1]) ? 1 : a1Counts[loc.name+loc.adminName1]+1;
                a3Counts[loc.name+loc.adminName3] = (!a3Counts[loc.name+loc.adminName3]) ? 1 : a3Counts[loc.name+loc.adminName3]+1;
                results.push(loc);
            });
            // append results to the model and build a distinct subtitle
            for(var x=0;x<results.length;x++) {
                location = results[x];
                location["areaLabel"] = buildAreaLabel(location, a1Counts ,a3Counts)
                citiesModel.append(location);
                noCityError.visible = false
            }
            //
            if (!citiesModel.count) {
                noCityError.visible = true
                noCityError.text = i18n.tr("No location was found for %1").arg(locationString.text)
            }
        } else {
            console.log("search: "+messageObject.error.msg+" / "+messageObject.error.request.url)
            httpFailedSearch.show();
        }
    }

    function buildAreaLabel(loc, a1Counts, a3Counts) {
        var label = "";
        label += ((loc.adminName1) ? loc.adminName1.replace(/ Region$/,'')+", ":"");
        label += ((loc.adminName2 && a1Counts[loc.name+loc.adminName1] > 1 && a3Counts[loc.name+loc.adminName3] > 1) ? loc.adminName2.replace(/ Region$/,'')+", ":"");
        label += ((loc.adminName3 && a1Counts[loc.name+loc.adminName1] > 1) ? loc.adminName3.replace(/ Region$/,'')+", ":"");
        label += ((Countries.codes[loc.country]) ? i18n.tr(Countries.codes[loc.country]): loc.country);
        return label;
    }

    function clear() {
        locationString.text = '';
        citiesModel.clear();
        listView.visible = true;
    }

    function doSearch() {
        if(locationString.text !== "") {
            cityList.visible = true
            citiesModel.clear();
            searching.running = true;
            WeatherApi.sendRequest({
                                       action: "searchByName",
                                       params: {name:locationString.text, units:"metric"}
                                   }, searchResponseHandler)
        }
    }

    SplashComponent {
        id: httpFailedSearch
        objectName: "HTTPFailedSearch"
    }

    ListModel {
        id: citiesModel
    }

    Rectangle {
        anchors.fill:parent
        color: "lightgrey"

        Item {
            anchors {
                top:parent.top
                topMargin:headerHeight+units.gu(1)
                margins: units.gu(2)
                right: parent.right
                left: parent.left
                bottom: parent.bottom
                bottomMargin:bottomHeight
            }
            Rectangle {
                id: searchInput
                height:units.gu(5)
                anchors {
                    right: parent.right
                    rightMargin: units.gu(1)
                    left: parent.left
                    leftMargin: units.gu(1)
                }
                color: "transparent"
                TextField {
                    id: locationString
                    objectName: "SearchField"
                    anchors.fill: parent
                    placeholderText: i18n.tr("Enter a city name")
                    hasClearButton: true
                    inputMethodHints: Qt.ImhNoPredictiveText
                    onAccepted: {
                        doSearch();
                        search_timer.stop()
                        locationString.focus = false
                    }
                    primaryItem: Image {
                        height: searchInput.height*0.5
                        width: searchInput.height*0.5
                        anchors.verticalCenter: parent.verticalCenter
                        anchors.verticalCenterOffset: -units.gu(0.2)
                        source: Qt.resolvedUrl("image://theme/search")
                    }

                    Timer {
                        id: search_timer

                        interval: 1100
                        repeat: false
                        onTriggered: {
                            if (locationString.text != "") {
                                doSearch();
                            }
                        }
                    }

                    onTextChanged: search_timer.restart()
                }
                ActivityIndicator {
                    id: searching
                    objectName: "SearchingSpinner"
                    anchors {
                        right: locationString.right
                        rightMargin: units.gu(2)
                        verticalCenter: locationString.verticalCenter
                    }
                }
            }

            Rectangle {
                id: cityList;
                anchors {
                    top:searchInput.bottom
                    left: parent.left
                    right: parent.right
                }
                height: addLocationPage.height-searchInput.height-units.gu(9.5)
                color: "transparent"
                visible: true
                Label {
                    id: noCityError
                    objectName: "noCityError"
                    visible: false
                    anchors {
                        fill: parent
                        leftMargin: units.gu(3)
                        rightMargin: units.gu(3)
                    }
                    fontSize: "medium"
                    wrapMode: Text.WordWrap
                    verticalAlignment: Text.AlignVCenter
                    horizontalAlignment: Text.AlignHCenter
                }
                Component {
                    id: sectionHeading
                    ListItem.Header {
                        Label {
                            text: section
                            anchors {
                                verticalCenter: parent.verticalCenter
                                left: parent.left
                                leftMargin: units.gu(1)
                            }
                            fontSize: "medium"
                        }
                    }
                }
                ListView {
                    id: listView;
                    objectName: "SearchResultList"
                    clip: true;
                    anchors.fill: parent;
                    model:  citiesModel;
                    section.property: "name"
                    section.criteria: ViewSection.FirstCharacter
                    section.delegate: sectionHeading;
                    section.labelPositioning: ViewSection.InlineLabels
                    delegate: ListItem.Standard {
                        Label {
                            id: searchResultName
                            objectName: "searchResult" + index
                            text: name
                            elide: Text.ElideRight
                            anchors {
                                top: parent.top
                                topMargin: units.gu(0.5)
                                left: parent.left
                                leftMargin: units.gu(3)
                                right: parent.right
                                rightMargin: units.gu(2)
                            }
                            fontSize: "large"
                        }
                        Label {
                            text: areaLabel
                            elide: Text.ElideRight
                            anchors {
                                top: searchResultName.bottom
                                bottom: parent.bottom
                                left: parent.left
                                leftMargin: units.gu(3)
                                right: parent.right
                                rightMargin: units.gu(2)
                            }
                            fontSize: "small"
                        }
                        onClicked: {
                            var location = citiesModel.get(index)
                            locationManagerPage.addLocation(location)
                            pageStack.pop()
                            clear()
                        }
                    }
                    Scrollbar {
                        flickableItem: listView;
                        align: Qt.AlignTrailing;
                    }
                }
            }
        }
    }

}
