/*
 * Copyright (C) 2013 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Raúl Yeguas <neokore@gmail.com>
 *              Martin Borho <martin@borho.net>
 */
import QtQuick 2.3
import Ubuntu.Components 1.1
import Ubuntu.Layouts 1.0
import "./" as Components

Rectangle {
    id: currentWeather
    color: "transparent"

    // For Status I take the same condition codes from OpenWeatherMap (http://openweathermap.org/wiki/API/Weather_Condition_Codes)
    property string icon
    property string currentTemp
    property string minTemp
    property string maxTemp
    property bool showMaxTemp
    property int windSpeed
    property string windDir
    property int humidity
    property real precipitation
    property string propPrecip
    property string precipType
    property string uvIndex
    property int pressure
    property variant hourly
    property string tempScale: (mainView.settings["units"] === "imperial") ? "F" : "C"
    property string speedScale: (mainView.settings["wind_units"] === "mph") ? "mph" : "km/h"
    property string precipScale: (mainView.settings["precip_units"] === "in") ? "in" : "mm"
    // Note that this should actually be a phone width, but since
    // the tablet layout is not yet working well on a N7, we
    // set the width to trigger this layout to be higher than that
    // of a N7, so that on that device it will still be in phone
    // mode
    // Nexus 7's width is 1920px/21px/GU = 91.43 GU (we use 95)
    // Nexus 7's width with new GU 1920px/18px/GU =  106.67 GU (we use 110)
    property real wideModeThreshold: units.gu(110)

    property NumberAnimation appearAnimation: frontrect.conditionIcon.appearAnimation

    anchors.fill: parent
    anchors.horizontalCenter: parent.horizontalCenter

    function getPrecipLabel() {
        // show the right Label, depending on precipitation type
        return (precipType === "snow") ? i18n.tr("Snow") : i18n.tr("Rain")
    }

    function getUvLabel() {
        var label = "",
            uvInt = parseInt(uvIndex);
        if(uvInt <= 2) {
            label = i18n.tr("Low");
        } else if(uvInt <= 5) {
            label = i18n.tr("Moderate");
        } else if(uvInt <= 8) {
            label = i18n.tr("High");
        } else if(uvInt <= 10) {
            label = i18n.tr("Very high");
        } else {
            label = i18n.tr("Extreme");
        }
        return label;
    }

    function resetFront() {
        // make front view visible
        frontrect.opacity = 1;
        frontrect.conditionIcon.opacity = 1;
        frontrect.conditionTempContent.opacity = 1;
        frontrect.condition.anchors.verticalCenterOffset = 0;
        frontrect.conditionTemp.anchors.verticalCenterOffset = 0;
    }

    Connections {
        target: dailyForecastItem
        onHideDetail: {flipable.flipped = false}
        onToggle: {flipable.flipped = !flipable.flipped}
    }

    Layouts {
        objectName: "WeatherLayout"
        id: layoutsCurrent
        anchors.fill: parent

        onCurrentLayoutChanged: {
            if(currentLayout) {
                // assure front shapes are visible
                currentWeather.resetFront()
            } else if(!flipable.state) {
                // when switching back to non wide mode, set to front state
                flipable.flipped = false
            }
        }

        layouts: [
            ConditionalLayout {
                name: "WideTablet"
                when: layouts.width > wideModeThreshold
                Rectangle {
                    color: "transparent"
                    anchors.fill: parent
                    ItemLayout {
                        item: "front"
                        id: frontConditional
                        opacity: 1
                         visible: true
                         anchors {
                             horizontalCenter: parent.horizontalCenter
                             verticalCenter: parent.verticalCenter
                             verticalCenterOffset: -units.gu(7.5)
                         }
                     }
                     ItemLayout {
                         item: "back"
                         height: layouts.height/2
                         opacity: 1
                         visible: true
                         anchors {
                             verticalCenter: parent.verticalCenter
                             verticalCenterOffset: units.gu(14.5)
                             horizontalCenter: parent.horizontalCenter
                         }
                     }                     
                }
            }
        ]
        Rectangle {
            id: flipable
            anchors.fill: parent
            color: "transparent"
            property bool flipped: false

            Components.ScrollingArea {}

            CurrentWeatherDetail {
                id: backrect
                Layouts.item: "back"
                anchors.fill: parent
                gridColumns: (layoutsCurrent.currentLayout == "WideTablet") ? 4 : 2
                opacity: (layoutsCurrent.currentLayout == "WideTablet") ? 1 : 0
                shapeSize: (layoutsCurrent.currentLayout == "WideTablet") ? units.gu(14) : units.gu(15)
            }

            CurrentWeatherFront {
                id: frontrect
                anchors.fill: parent
                Layouts.item: "front"
                viewMode: (layoutsCurrent.currentLayout == "WideTablet") ? 1 : 0
            }

            states:[
                State {
                    name: "back"
                    when: flipable.flipped && layouts.width < wideModeThreshold
                },
                State {
                    name: "front"
                    when: !flipable.flipped && layouts.width < wideModeThreshold
                }
            ]

            transitions:[
                Transition {
                    to:"back"
                    SequentialAnimation{
                        ParallelAnimation {
                            NumberAnimation { target: frontrect.condition; property: "anchors.verticalCenterOffset"; duration: UbuntuAnimation.SlowDuration; to: units.gu(8); easing: UbuntuAnimation.StandardEasing }
                            NumberAnimation { target: frontrect.conditionTemp; property: "anchors.verticalCenterOffset"; duration: UbuntuAnimation.SlowDuration; to: units.gu(-8); easing: UbuntuAnimation.StandardEasing }
                            NumberAnimation {target: frontrect.conditionIcon; property: "opacity"; to:0; duration: UbuntuAnimation.SlowDuration}
                            NumberAnimation {target: frontrect.conditionTempContent; property: "opacity"; to:0; duration: UbuntuAnimation.SlowDuration}
                        }
                        ParallelAnimation {
                            NumberAnimation {target: backrect; property: "opacity"; to:1; duration: UbuntuAnimation.SlowDuration}
                            NumberAnimation {target: frontrect; property: "opacity"; to:0; duration: UbuntuAnimation.SlowDuration}
                        }
                    }
                },
                Transition {
                    from: "back"
                    to:"front"
                    ScriptAction{
                        script: {
                            // show front weather when switching to new day (by keyboard)
                            if(dailyForecastList.currentItem == dailyForecastItem){
                                 back2frontAnim.start();
                            } else {
                                backrect.opacity = 0;
                                currentWeather.resetFront()
                            }
                        }
                    }
                }
            ]

            SequentialAnimation{
                id: back2frontAnim
                running: false
                ParallelAnimation {
                    NumberAnimation {target: frontrect; property: "opacity"; to:1; duration: UbuntuAnimation.SlowDuration}
                    NumberAnimation {target: backrect; property: "opacity"; to:0; duration: UbuntuAnimation.SlowDuration}
                }
                ParallelAnimation{
                    NumberAnimation {target: frontrect.conditionIcon; property: "opacity"; to:1; duration: UbuntuAnimation.SlowDuration}
                    NumberAnimation {target: frontrect.conditionTempContent; property: "opacity"; to:1; duration: UbuntuAnimation.SlowDuration}
                    NumberAnimation { target: frontrect.condition; property: "anchors.verticalCenterOffset"; duration: UbuntuAnimation.SlowDuration; to: 0; easing: UbuntuAnimation.StandardEasingReverse }
                    NumberAnimation { target: frontrect.conditionTemp; property: "anchors.verticalCenterOffset"; duration: UbuntuAnimation.SlowDuration; to: 0; easing: UbuntuAnimation.StandardEasingReverse }
                }
            }
        }
    }
}
