/*
 * Copyright (C) 2013,2014 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Martin Borho <martin@borho.net>
                Raúl Yeguas <neokore@gmail.com>
                Riccardo Padovani <ricki.padovani@gmail.com>
 */
import QtQuick 2.3
import Ubuntu.Components 1.1
import Ubuntu.Components.Popups 1.0
import Ubuntu.Components.ListItems 1.0 as ListItem
import "CountryCodes.js" as Countries
import "WeatherApi.js" as WeatherApi

Page {
    id: locationManagerPage
    objectName: "LocationManagerPage"
    title: i18n.tr("Edit locations")
    anchors.fill: parent
    visible: false

    readonly property real headerHeight: units.gu(9.5)
    readonly property real bottomHeight: units.gu(9.5)

    property int initial_sum: 0
    property bool limited: (locationModel.count > 7)
    property bool locationsChanged: false

    head.backAction: Action {
        iconName: "back"
        onTriggered: {
            if(locationModel.count > 0) {
                popFromStack()
            }
        }
    }

    Component.onCompleted: {        
        loadData();
    }

    Connections { target: mainView; onEscapeKey: popFromStack()}

    function popFromStack() {
        pageStack.pop()
        // refresh at least from storage, to prevent header content from disapearing
        refreshData(!locationsChanged, locationsChanged);
    }

    function lookupResponseHandler(messageObject) {
        lookupIndicator.running = false;
        if(!messageObject.error) {
            if(messageObject.result.locations && messageObject.result.locations.length > 0) {
                var location = messageObject.result.locations[0];
                currentLocationItem.detectedLocation = location;
                currentLocationLabel.text = location.name
                currentLocationAreaLabel.text = location.adminName1
                locationLookupItem.visible = false
                currentLocationItem.visible = true;
            }
        } else {
            console.log("search: "+messageObject.error.msg+" / "+messageObject.error.request.url)
            httpFailedLookup.show();
        }
    }

    function loadData() {
        storage.getLocations(fillLocationsList);
    }

    function checkLocationExists(location) {
        for(var x=0;x<locationModel.count;x++) {
            var loc = locationModel.get(x);
            if(loc.services.geonames && (loc.services.geonames === location.services.geonames)) {
                existsNotification.show()
                return true;
            }
        }
        return false;
    }

    function fillLocationsList(locations) {
        initial_sum = locations.length;
        locationModel.clear()
        for(var x=0;x<locations.length;x++) {
            var dbId = locations[x].db.id,
                    location = locations[x].location;
            // add db-id to the location object
            location.dbId = dbId;
            locationModel.append(locations[x].location);
        }
    }

    function addLocation(location) {
        if(!checkLocationExists(location)) {
            if(location.dbId === undefined || location.dbId=== 0) {                
                var newDbId = storage.insertLocation({location:location});
                location.dbId = parseInt(newDbId);
                locationModel.append(location)
                locationsChanged = true;
            }
        }
    }

    function deleteLocation(index) {
        var location = locationModel.get(index),
            locDbId = location.dbId;
        locationModel.remove(index)
        storage.clearLocation(locDbId);
        locationsChanged = true;
    }

    Dialog {
        id:existsNotification
        title: i18n.tr("Location already added.")
        Button {
            text: i18n.tr("OK")
            onClicked: PopupUtils.close(existsNotification)
        }
    }

    ListModel {
        id: locationModel
    }

    Rectangle {
        anchors.fill:parent
        color: "lightgrey"

        Column {
            anchors {
                top:parent.top
                topMargin:headerHeight
                margins: units.gu(2)
                right: parent.right
                left: parent.left
                bottom: parent.bottom
                bottomMargin:bottomHeight
            }
            Rectangle {
                anchors {
                    left:parent.left
                    right: parent.right
                }
                height: units.gu(5)
                color: "transparent"
                Label {
                    text: " "+i18n.tr("Current location")
                    anchors {
                        verticalCenter: parent.verticalCenter
                        left: parent.left
                        leftMargin: units.gu(1)
                    }
                    fontSize: "medium"
                }
            }
            Rectangle {
                color: "#A9A9A9"
                anchors {
                    left:parent.left
                    right: parent.right
                }
                height: units.gu(0.1)
            }
            ListItem.Standard {
                id: locationLookupItem
                objectName: "LocationLookupItem"
                progression: true
                opacity: (!limited) ? 1 : 0.5
                enabled: !limited
                Label {
                    id: locationLookupItemLabel
                    text: i18n.tr("Scan for location")
                    visible: !lookupIndicator.running
                    anchors {
                        verticalCenter: parent.verticalCenter
                        left: parent.left
                        leftMargin: units.gu(3)
                    }
                    fontSize: "large"
                }
                ActivityIndicator {
                    id: lookupIndicator
                    objectName: "LookupIndicator"
                    running:false
                    anchors.verticalCenter: parent.verticalCenter
                    anchors.left: parent.left
                    anchors.leftMargin: units.gu(3)
                }
                onClicked: {
                    lookupIndicator.running = true;
                    WeatherApi.sendRequest({
                                               action: "getGeoIp",
                                               params: {}
                                           }, lookupResponseHandler);
                }
            }
            ListItem.Standard {
                id: currentLocationItem
                objectName: "CurrentLocationItem"
                property var detectedLocation: null;
                visible: false
                opacity: (!limited) ? 1 : 0.5
                enabled: !limited
                Label {
                    id: currentLocationLabel
                    objectName: "CurrentLocationLabel"
                    text: ""
                    elide: Text.ElideRight
                    anchors {
                        top: parent.top
                        topMargin: units.gu(0.5)
                        left: parent.left
                        leftMargin: units.gu(3)
                        right:parent.right
                        rightMargin: lookupItemAddButton.width+units.gu(5.5)
                    }
                    fontSize: "large"
                }
                Label {
                    id: currentLocationAreaLabel
                    text: ""
                    elide: Text.ElideRight
                    anchors {
                        top: currentLocationLabel.bottom
                        bottom: parent.bottom
                        left: parent.left
                        leftMargin: units.gu(3)
                        right:parent.right
                        rightMargin: lookupItemAddButton.width+units.gu(5.5)
                    }
                    fontSize: "small"
                }
                control: Button {
                    objectName: "LookupItemAddButton"
                    id: lookupItemAddButton
                    anchors.verticalCenter: parent.verticalCenter
                    // TRANSLATORS: this is the button to add a location (city) that has been
                    // detected to be the current position
                    text: i18n.tr("Add")
                    MouseArea {
                        anchors.fill: parent
                        onClicked: {
                            if(currentLocationItem.detectedLocation) {
                                addLocation(currentLocationItem.detectedLocation)
                            }
                            lookupItemAddButton.visible = false;
                        }
                    }
                }
            }
            Rectangle {
                anchors {
                    left:parent.left
                    right: parent.right
                }
                height: units.gu(5)
                color: "transparent"
                Label {
                    text: " "+i18n.tr("Locations")
                    anchors {
                        verticalCenter: parent.verticalCenter
                        left: parent.left
                        leftMargin: units.gu(1)
                    }
                    fontSize: "medium"
                }
            }
            Rectangle {
                color: "#A9A9A9"
                anchors {
                    left:parent.left
                    right: parent.right
                }
                height: units.gu(0.1)
            }
            Rectangle {
                anchors {
                    left:parent.left
                    right: parent.right
                }
                clip:true
                color:"transparent"
                height: locationManagerPage.height-locationLookupItem.height-units.gu(19.2)
                ListView {
                    objectName: "LocationList"
                    anchors.fill: parent
                    model: locationModel
                    delegate: ListItem.Standard {
                        objectName: "existingLocationItem" + index
                        Label {
                            id: searchResultName
                            objectName: "existingLocation" + index
                            text: name
                            elide: Text.ElideRight
                            anchors {
                                top: parent.top
                                topMargin: units.gu(0.5)
                                left: parent.left
                                leftMargin: units.gu(3)
                                right:parent.right
                                rightMargin: units.gu(3)
                            }
                            fontSize: "large"
                        }
                        Label {
                            text: adminName1.replace(/ Region$/,'')+", "+((Countries.codes[country]) ? i18n.tr(Countries.codes[country]): country)
                            elide: Text.ElideRight
                            anchors {
                                top: searchResultName.bottom
                                bottom: parent.bottom
                                left: parent.left
                                leftMargin: units.gu(3)
                                right:parent.right
                                rightMargin: units.gu(3)
                            }
                            fontSize: "small"
                        }
                        removable: true
                        onItemRemoved: {
                            deleteLocation(index);
                        }
                    }
                    footer: ListItem.Standard {
                        objectName: "AddCityListItem"
                        id: addCityListItem
                        progression: true
                        opacity: (!limited) ? 1 : 0.5
                        enabled: !limited
                        onEnabledChanged: {
                            if(enabled) {
                                footerTitle.anchors.verticalCenterOffset = 0
                            } else {
                                footerTitle.anchors.verticalCenterOffset = -units.gu(1)
                            }
                        }
                        Label {
                            id: footerTitle
                            text: i18n.tr("Add city")
                            anchors {
                                verticalCenter: parent.verticalCenter
                                left: parent.left
                                leftMargin: units.gu(3)
                                right:parent.right
                                rightMargin: units.gu(3)
                            }
                            fontSize: "large"
                        }
                        Label {
                            visible: limited
                            text: i18n.tr("Remove a city first.")
                            elide: Text.ElideRight
                            anchors {
                                top: footerTitle.bottom
                                left: parent.left
                                leftMargin: units.gu(3)
                                right:parent.right
                                rightMargin: units.gu(3)
                            }
                            fontSize: "small"
                        }
                        onClicked: {
                            pageStack.push(Qt.createComponent("AddLocationPage.qml"))
                        }
                    }
                }
            }
        }
    }

    SplashComponent {
        id: httpFailedLookup
        objectName: "HTTPFailedLookup"
        message: i18n.tr("Couldn't specify your current location, please try later again!")
    }
}
