/*
 * Copyright (C) 2013 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Raúl Yeguas <neokore@gmail.com>
 *              Martin Borho <martin@borho.net>
 */
import QtQuick 2.3
import Ubuntu.Components.Popups 1.0
import Ubuntu.Components 1.1
import Ubuntu.Components.ListItems 1.0 as ListItem
import Ubuntu.Layouts 1.0
import QtFeedback 5.0

Tab {
    id: locationTab
    objectName: "LocationTab" + index

    anchors.fill: parent

    property int locationIndex: 0
    property var locationData: {}

    // signals
    signal toggle()
    onToggle: dailyForecastList.currentItem.toggle()

    signal addLocation()
    onAddLocation: editLocationAction.trigger()

    signal toDayBefore()
    onToDayBefore: {
        dailyForecastList.currentItem.hideDetail();
        dailyForecastList.decrementCurrentIndex();
    }

    signal toNextDay()
    onToNextDay: {
        dailyForecastList.currentItem.hideDetail();
        dailyForecastList.incrementCurrentIndex();
    }


    function formatTimestamp(dateData, format) {
        var date = new Date(dateData.year, dateData.month, dateData.date, dateData.hours, dateData.minutes)
        return Qt.formatDate(date, i18n.tr(format))
    }

    onVisibleChanged: {
        if(dailyForecastList.currentIndex > -1) {
            dailyForecastList.currentIndex = 0;
            if(visible == true) {
                var data = dayForecastModel.get(0)
                adjustBackground(Math.round(data.temp))
                dailyForecastList.currentItem.appearAnimation.start()
            }
        }
    }

    function ready() {
        var locData = mainView.locationsList[locationIndex],
            units = (mainView.settings.units === 'imperial') ? 'imperial' : 'metric',
            wind_units = (mainView.settings.wind_units === 'mph') ? 'imperial' : 'metric',
            precip_units = (mainView.settings.precip_units === 'in') ? 'imperial' : 'metric',
            dailyForecasts = locData.data,
            dailyLength = (dailyForecasts) ? dailyForecasts.length : 0;

        // set location data as property
        locationData = locData;

        // set last updated label
        lastUpdatedComponent.updateDate = (locationData.updated) ? locationData.updated : new Date().getTime()

        // set daily forecasts
        if(dailyLength > 0) {
            noDataAvaible.visible = false;
            for(var x=0;x<dailyLength;x++) {
                //
                dayForecastModel.append({
                    dateRel: "",//Tomorrow",
                    date: formatTimestamp(dailyForecasts[x].date, 'dddd, dd MMMM'),
                    temp: (dailyForecasts[x]["current"] === undefined) ? dailyForecasts[x][units].tempMax :
                        dailyForecasts[x]["current"][units].temp,
                    tempMin: dailyForecasts[x][units].tempMin.toString(),
                    tempMax: (dailyForecasts[x]["current"] !== undefined
                        && dailyForecasts[x][units].tempMax !== undefined) ? dailyForecasts[x][units].tempMax.toString() : "",
                    condIcon: (dailyForecasts[x]["current"] !== undefined &&
                               dailyForecasts[x]["current"].icon !== undefined) ? dailyForecasts[x]["current"].icon : dailyForecasts[x].icon,
                    wind_speed: dailyForecasts[x][wind_units].windSpeed,
                    wind_dir: dailyForecasts[x].windDir,
                    humid: dailyForecasts[x].humidity,
                    precip: +(dailyForecasts[x][precip_units].rain || dailyForecasts[x][precip_units].snow || 0).toFixed(2),
                    prop_precip: ((dailyForecasts[x].propPrecip !== undefined  && ""+dailyForecasts[x].propPrecip) || ""),
                    precip_type: dailyForecasts[x].precipType || "",
                    uv: (dailyForecasts[x].uv !== undefined) ? dailyForecasts[x].uv.toString() : "",
                    press: dailyForecasts[x].pressure || false,
                    hours: dailyForecasts[x]["hourly"]
                });
            }
        } else {
            noDataAvaible.visible = true;
        }
    }

    HapticsEffect {
        id: hourlyScrollEffect
        attackTime: 1
        intensity: 0.2
        duration: 5
        fadeTime: 1
    }

    // Menu for options
    page: Page {
        id: locationPage
        isLeaf: true
        anchors.fill: parent
        anchors.topMargin: units.gu(9.5)
        head.foregroundColor: "#FFFFFF"
        head.actions: [
            Action {
                id: refreshAction
                objectName: "RefreshButton"
                iconName: "reload"
                text: i18n.tr("Refresh")

                onTriggered: {
                    mainView.tabIndexAtRefresh = tabs.selectedTabIndex
                    mainView.refreshData(false, true);
                }
            },
            Action {
                objectName: "EditButton"
                id: editLocationAction
                iconName: "add"
                text: i18n.tr("Locations")

                onTriggered: {
                    mainView.showLocationManager()
                }
            },
            Action {
                objectName: "SettingsButton"
                id: configAction
                iconName: "settings"
                text: i18n.tr("Settings")

                onTriggered: {
                    mainView.showSettings();
                }
            }
        ]

        ListModel {
            id: dayForecastModel
        }

        Layouts {
            objectName: "WeatherLayout"
            id: layouts
            anchors.fill: parent
            layouts: [
                ConditionalLayout {
                    name: "WeatherData"
                    when: layouts.width > units.gu(80)
                    Rectangle {
                        anchors.fill: parent
                        color: "transparent"
                        ItemLayout {
                            item: "ForecastList"
                            width: layouts.width*0.625
                            anchors {
                                top: parent.top
                                bottom: parent.bottom
                            }
                        }
                        Rectangle {
                            width: layouts.width*0.375
                            anchors {
                                top: parent.top
                                bottom:parent.bottom
                                right: parent.right
                            }
                            color: "black"
                            opacity: 0.3
                        }
                        ItemLayout {
                            item: "TabFooter"
                            // We're making the footer to be full-width
                            // so that it looks good on a Nexus 7
                            height: units.gu(8)
                            anchors {
                                bottom: parent.bottom
                                left: parent.left
                                right: parent.right
                            }
                        }
                        Rectangle {
                            id: sideLabel
                            objectName: "SideLabel"
                            anchors.top: parent.top
                            anchors.right: parent.right
                            width: layouts.width*0.375
                            height: units.gu(6)
                            color: "transparent"
                            Label {
                                text: i18n.tr("Forecast")
                                fontSize: "large"
                                anchors {
                                    left:parent.left
                                    leftMargin: units.gu(3)
                                    top: parent.top;
                                    bottom: parent.bottom
                                }
                                verticalAlignment: Text.AlignVCenter
                            }
                        }
                        ListView {
                            id: locationDailySide
                            width: (layouts.width*0.375)-units.gu(1)
                            height: layouts.height-sideLabel.height-units.gu(9)
                            anchors.right: parent.right
                            anchors.top: sideLabel.bottom
                            model:dayForecastModel
                            clip: true
                            delegate: ListItem.Subtitled {
                                subText:  "H:"+Math.round(temp)+"  L:"+Math.round(tempMin)
                                text: date
                                //selected: (dailyForecastList.currentIndex == index)
                                iconSource: "../resources/images/"+dayForecastModel.get(index).condIcon+".png"
                                onClicked: { dailyForecastList.currentIndex = index }
                            }
                        }
                    }
                }
            ]

            ListView {
                id: dailyForecastList
                objectName: "DailyForecastList"
                anchors.fill:parent
                model:dayForecastModel
                Layouts.item: "ForecastList"
                onCurrentIndexChanged: {
                    if(visible) {
                        var data = dayForecastModel.get(currentIndex)
                        adjustBackground(Math.round(data.temp))
                        dailyForecastList.currentItem.appearAnimation.start()
                    }
                }

                interactive: false
                snapMode: ListView.SnapOneItem
                highlightMoveDuration: 150
                highlightRangeMode: ListView.StrictlyEnforceRange
                verticalLayoutDirection: ListView.TopToBottom
                flickableDirection: Flickable.VerticalFlick
                clip: true

                delegate: Item {
                    id: dailyForecastItem
                    anchors {
                        left:parent.left
                        right: parent.right
                    }
                    height: locationPage.height

                    property NumberAnimation appearAnimation: currentWeatherComponent.appearAnimation

                    signal hideDetail()
                    signal toggle()

                    Rectangle {
                        id: listRectangle
                        height: dailyForecastList.height-tabFooter.height
                        anchors {
                            left:parent.left
                            right: parent.right
                            top: parent.top
                        }
                        color: "transparent"

                        DateComponent {
                            id: dateComponent
                            dateRelative: dateRel
                            dateString: date
                        }
                        CurrentWeather {
                            id: currentWeatherComponent
                            currentTemp: temp
                            minTemp: tempMin
                            maxTemp: tempMax
                            icon: condIcon
                            windSpeed: wind_speed
                            windDir: wind_dir
                            humidity: humid
                            precipitation: precip
                            propPrecip: prop_precip
                            precipType: precip_type
                            pressure: press
                            uvIndex: uv
                            hourly: hours
                            anchors.top: dateComponent.bottom
                            showMaxTemp: (index === 0)
                        }
                    }
                }
                LastUpdatedComponent{
                    id: lastUpdatedComponent
                    anchors.bottom: parent.bottom
                    anchors.bottomMargin: (tabFooter.visible) ? units.gu(8) : 0
                }
                TabFooter {
                    id:tabFooter
                    objectName: "TabFooter"
                    Layouts.item: "TabFooter"
                    visible: !dataProviderLogo.visible
                }
                Image {
                    id: dataProviderLogo
                    objectName: "DataProviderLogo"
                    visible: (settings["service"] === "openweathermap")
                    source: Qt.resolvedUrl("../resources/images/openWeatherMapLogo.png")
                    anchors {
                        bottom: parent.bottom
                        right: parent.right
                        bottomMargin: units.gu(1)
                        rightMargin: units.gu(1)
                    }
                    height: units.gu(8)
                    width: units.gu(8)
                    fillMode: Image.PreserveAspectFit
                    MouseArea {
                        anchors.fill: parent
                        onClicked: Qt.openUrlExternally("http://openweathermap.org/city/" + locationData.location.services.openweathermap)
                    }
                }
                Rectangle {
                    id: noDataAvaible
                    visible: false
                    anchors {
                        left: parent.left
                        leftMargin: units.gu(5)
                        right: parent.right
                        rightMargin: units.gu(5)
                        verticalCenter: parent.verticalCenter
                        verticalCenterOffset: -units.gu(10)
                    }
                    color: "transparent"
                    Label {
                        anchors.fill:parent
                        text: i18n.tr("No weather data available at the moment, please try to refresh later again!")
                        fontSize: "large"
                        wrapMode: Text.WordWrap
                    }
                }
            }
        }
    }
}
