/*
 * Copyright (C) 2013,2014 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Raúl Yeguas <neokore@gmail.com>
 *              Martin Borho <martin@borho.net>
 */

import QtQuick 2.3

MouseArea {
    id: flippedarea
    property bool wasHourlyScrolled: false
    property int startY: 0
    property double startTime: 0
    property int sectionHeight: flippedarea.height/hourly.count
    property int lastIndex: -1
    anchors.fill: parent
    anchors.top: parent.top
    anchors.topMargin: units.gu(1.2)
    onClicked: {
        mouse.accepted = true;
        if(!wasHourlyScrolled) {
            var movement = Math.round(startY-mouse.y);
            if(movement < 10 && movement > -10) {
                // flip to detail data
                flipable.flipped = !flipable.flipped;
            } else {
                // switch to next/prevoius day
                if(movement >= 10) {
                    dailyForecastList.incrementCurrentIndex();
                } else if(movement <= -10) {
                    dailyForecastList.decrementCurrentIndex();
                }
                dailyForecastList.positionViewAtIndex(dailyForecastList.currentIndex,ListView.SnapPosition);
                flipable.flipped = false;
            }
        }
    }
    onPressed: {
        wasHourlyScrolled = false;
        startY = mouse.y;
        startTime = new Date().getTime()
    }
    onPositionChanged: {
        mouse.accepted = true;
        var distance = (flippedarea.height - mouse.y),
            now = new Date().getTime(),
            speed = distance/(now-startTime);
        if(speed < 2.3) {
            // slow scrolling detected
            var movement = Math.round(distance/sectionHeight);
            if(movement > 0) {
                var hourIndex = movement-1,
                    currentHour = currentWeather.hourly.get(hourIndex);
                if( currentHour !== undefined && (lastIndex < 0 || lastIndex !== hourIndex)) {
                    wasHourlyScrolled = true;
                    frontrect.conditionIcon.condition = currentHour.icon;
                    var temp = currentHour[mainView.settings["units"]].temp;
                    frontrect.conditionTempContent.currentTemp = temp;
                    adjustBackground(Math.round(temp));
                    // adjust detail data
                    currentWeather.humidity = currentHour.humidity
                    currentWeather.windSpeed = currentHour[(mainView.settings.wind_units === 'mph') ? 'imperial' : 'metric'].windSpeed
                    currentWeather.windDir = currentHour.windDir
                    if(currentHour.propPrecip !== undefined && currentHour.propPrecip !== "") {
                        currentWeather.propPrecip = currentHour.propPrecip;
                    } else {
                        currentWeather.pressure = currentHour.pressure
                    }
                    if(currentHour.uvIndex !== undefined && currentHour.uvIndex !== "") {
                        currentWeather.uvIndex = currentHour.uvIndex;
                    } else {
                        var precip_units = (mainView.settings.precip_units === 'in') ? 'imperial' : 'metric';
                        currentWeather.precipitation = +(currentHour[precip_units].rain || currentHour[precip_units].snow || 0).toFixed(2)
                    }
                    // adjust date data
                    var dateData =  currentHour.date;
                    var dateTime = new Date(dateData.year, dateData.month, dateData.date, dateData.hours, dateData.minutes)
                    // TRANSLATORS: this is a date to which a time of the day will be appended
                    var dateTimeString = Qt.formatDateTime(dateTime, i18n.tr("dddd, dd MMMM")) + ' - ' + Qt.formatTime(dateTime);
                    dateComponent.dateString = dateTimeString;
                    dateComponent.renderText();
                    //
                    hourlyScrollEffect.start();
                    lastIndex = hourIndex;
                }
            }
        }
    }
    onReleased: {
        if(wasHourlyScrolled) {
            // reset temp and date to current day
            var currentDayData = dayForecastModel.get(dailyForecastList.currentIndex);
            frontrect.conditionIcon.condition = currentDayData.condIcon;
            frontrect.conditionIcon.appearAnimation.start();
            frontrect.conditionTempContent.currentTemp = currentDayData.temp;
            // set back detail data
            currentWeather.humidity = currentDayData.humid
            currentWeather.windSpeed = currentDayData.wind_speed
            currentWeather.windDir = currentDayData.wind_dir
            if(currentDayData.prop_precip !== undefined && currentDayData.prop_precip !== "") {
                currentWeather.propPrecip = currentDayData.prop_precip;
            } else {
                currentWeather.pressure = currentDayData.press
            }
            if(currentDayData.uv !== undefined && currentDayData.uv !== "") {
                currentWeather.uvIndex = currentDayData.uv;
            } else {
                var precip_units = (mainView.settings.precip_units === 'in') ? 'imperial' : 'metric';
                currentWeather.precipitation = currentDayData.precip
            }
            // set back date data
            adjustBackground(Math.round(currentDayData.temp));
            dateComponent.dateString = currentDayData.date;
            dateComponent.renderText();
            lastIndex = -1;
        }
    }
}
