/*
 * Copyright (C) 2013, 2014 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Raúl Yeguas <neokore@gmail.com>
 *              Martin Borho <martin@borho.net>
 */
import QtQuick 2.3
import Ubuntu.Components 1.1

Page {
    title: i18n.tr("Settings")
    anchors.fill: parent
    visible: false
    readonly property real headerHeight: units.gu(9.5)
    readonly property real bottomHeight: units.gu(9.5)
    property bool refresh_from_storage: false
    property bool refresh_from_service: false

    head.backAction: Action {
        iconName: "back"
        onTriggered: {
            if(refresh_from_storage === true || refresh_from_service === true) {
                storage.getSettings(function(storedSettings) {
                    for(var settingName in storedSettings) {
                        settings[settingName] = storedSettings[settingName];
                    }
                    refreshData(refresh_from_storage, refresh_from_service);
                });
            } else {
                // reload tabs from storage, to prevent header content from diappearing
                refreshData(true);
            }
            popFromStack()
        }
    }

    Connections { target: mainView; onEscapeKey: popFromStack()}

    function popFromStack() {
        pageStack.pop()
    }

    function fillTempSelector() {
        temperatureUnitsSelectorModel.append({ name: "celsiusOption", label: i18n.tr("Celsius")});
        temperatureUnitsSelectorModel.append({ name: "fahrenheitOption", label: i18n.tr("Fahrenheit")});
        temperatureUnitsSelector.selectedIndex = (settings["units"] === "imperial") ? 1 : 0;
    }

    function fillWindSelector() {
        windUnitsSelectorModel.append({ name: "kmhOption", label: i18n.tr("Kilometers per hour")});
        windUnitsSelectorModel.append({ name: "mphOption", label: i18n.tr("Miles per hour")});
        windUnitsSelector.selectedIndex = (settings["wind_units"] === "mph") ? 1 : 0;
    }

    function fillPrecipSelector() {
        precipitationUnitsSelectorModel.append({ name: "millimetersOption", label: i18n.tr("Millimeters")});
        precipitationUnitsSelectorModel.append({ name: "inchesOption", label: i18n.tr("Inches")});
        precipitationUnitsSelector.selectedIndex = (settings["precip_units"] === "in") ? 1 : 0;
    }

    Component.onCompleted: {
        refresh_from_service = false;
        refresh_from_storage = false;
        fillTempSelector();
        fillWindSelector();
        fillPrecipSelector();
        mainView.shortCutsLimited = true;
    }

    Rectangle {
        anchors.fill:parent
        color: "lightgrey"

        Column {
            anchors {
                top:parent.top
                topMargin:headerHeight+units.gu(3)
                margins: units.gu(2)
                right: parent.right
                left: parent.left
                bottom: parent.bottom
                bottomMargin:bottomHeight
            }


            Component {
                id: unitsSelectorDelegate
                OptionSelectorDelegate {
                    text: i18n.tr(label)
                    objectName: name
                }
            }

            ListModel {
                id: temperatureUnitsSelectorModel
            }

            OptionSelector {
                id: temperatureUnitsSelector
                objectName: "TemperatureUnitsSelector"
                text: i18n.tr("Temperature units")
                delegate: unitsSelectorDelegate
                model: temperatureUnitsSelectorModel
                onSelectedIndexChanged: {
                    var selectedUnit = (selectedIndex === 0) ? "metric" : "imperial";
                    storage.saveSetting("units", selectedUnit);
                    refresh_from_storage = true;
                }
            }

            ListModel {
                id: windUnitsSelectorModel
            }

            OptionSelector {
                id: windUnitsSelector
                objectName: "WindUnitsSelector"
                text: i18n.tr("Wind speed units")
                delegate: unitsSelectorDelegate
                model: windUnitsSelectorModel
                onSelectedIndexChanged: {
                    var selectedWindUnit = (selectedIndex === 0) ? "kmh" : "mph";
                    storage.saveSetting("wind_units", selectedWindUnit);
                    refresh_from_storage = true;
                }
            }

            ListModel {
                id: precipitationUnitsSelectorModel
            }

            OptionSelector {
                id: precipitationUnitsSelector
                objectName: "PrecipitationUnitsSelector"
                text: i18n.tr("Precipitation units")
                delegate: unitsSelectorDelegate
                model: precipitationUnitsSelectorModel
                // Precipitation units not available at TWC
                visible: (serviceSelector.selectedIndex !== 0) ?  1 : 0
                opacity: (serviceSelector.selectedIndex !== 0) ?  1 : 0
                Behavior on visible {
                    enabled: (serviceSelector.selectedIndex === 0)
                    NumberAnimation { duration:  1500}
                }
                Behavior on opacity {
                    NumberAnimation {
                        easing: UbuntuAnimation.StandardEasingReverse;
                        duration: UbuntuAnimation.SlowDuration
                    }
                }
                onSelectedIndexChanged: {
                    var selectedPrecipUnit = (selectedIndex === 0) ? "mm" : "in";
                    storage.saveSetting("precip_units", selectedPrecipUnit);
                    refresh_from_storage = true;
                }
            }

            ListModel {
                id: serviceSelectorModel
                ListElement { name: "twcOption"; label: "The Weather Channel" }
                ListElement { name: "owmOption"; label: "Openweathermap" }
            }

            OptionSelector {
                id: serviceSelector
                objectName: "ServiceSelector"
                text: i18n.tr("Weather Service")
                delegate: unitsSelectorDelegate
                model: serviceSelectorModel
                selectedIndex: (settings["service"] === "openweathermap") ? 1 : 0;
                onSelectedIndexChanged: {
                    var selectedService = (selectedIndex === 0) ? "weatherchannel" : "openweathermap";
                    storage.saveSetting("service", selectedService);
                    refresh_from_service = true;
                }
            }
        }
    }
}
