/*
 * Copyright (C) 2013, 2014 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Raúl Yeguas <neokore@gmail.com>
 *              Martin Borho <martin@borho.net>
 *              Andrew Starr-Bochicchio <a.starr.b@gmail.com>
 */
import QtQuick 2.3
import Ubuntu.Components 1.1
import "components" as Components
import "components/GradientsMap.js" as Gradients
import "components/WeatherApi.js" as WeatherApi
import "key.js" as Key
import Ubuntu.Components.Popups 1.0

MainView {
    // objectName for functional testing purposes (autopilot-qt5)
    objectName: "weather"
    applicationName: "com.ubuntu.weather"

    id: mainView
    
    width: units.gu(50)
    height: units.gu(67)

    useDeprecatedToolbar: false

    headerColor: "#E04414"
    backgroundColor: "#E04414"
    footerColor: "#E1983E"

    property string newHeaderColor: "#E04414"
    property string newBackgroundColor: "#E04414"
    property string newFooterColor: "#E1983E"

    property bool wideAspect: width >= units.gu(80)

    property var locationsList: []
    property int tabIndexAtRefresh: -1
    property bool shortCutsLimited: false

    // some signals
    signal escapeKey()

    // keyboard shortcuts
    focus: true
    Keys.onPressed: {
        if(!shortCutsLimited && tabs.count > 0 && event.modifiers >= 0 && Qt.ControlModifier) {
            switch(event.key) {
                case Qt.Key_Tab: // CRTL-TAB
                    tabs.selectedTabIndex = (tabs.count!==tabs.selectedTabIndex+1) ? tabs.selectedTabIndex+1 : 0;
                    break;
                case Qt.Key_Backtab: // CRTL-ALT-TAB
                    tabs.selectedTabIndex = (tabs.selectedTabIndex>0) ? tabs.selectedTabIndex-1 : tabs.count-1;
                    break;
                case Qt.Key_N:// CRTL-N
                    mainView.showLocationManager();
                    break;
                case Qt.Key_R:// CRTL-R
                    refreshData(false, true);
                    break;
            }
        }
        event.accepted = false;
    }
    Keys.onSpacePressed: {
        if(!shortCutsLimited && tabs)
            tabs.tabChildren[tabs.selectedTabIndex].toggle();
        event.accepted = false;
    }
    Keys.onUpPressed: {
        if(!shortCutsLimited && tabs)
            tabs.tabChildren[tabs.selectedTabIndex].toDayBefore();
        event.accepted = false;
    }
    Keys.onDownPressed: {
        if(!shortCutsLimited && tabs)
            tabs.tabChildren[tabs.selectedTabIndex].toNextDay();
        event.accepted = false;
    }
    Keys.onEscapePressed: {
        if(tabs) escapeKey();
        event.accepted = false;
    }

    // set default values for settings here
    property var settings: {
        "units": Qt.locale().measurementSystem === Locale.MetricSystem ? "metric" : "imperial",
        "wind_units": Qt.locale().measurementSystem === Locale.MetricSystem ? "kmh" : "mph",
        "precip_units": Qt.locale().measurementSystem === Locale.MetricSystem ? "mm" : "in",
        "service": "weatherchannel"
    }

    ParallelAnimation {
        id: gradientAnimation
        ColorAnimation {
            target: mainView
            property: "headerColor"
            from: headerColor
            to: newHeaderColor
            duration:UbuntuAnimation.BriskDuration
            easing: UbuntuAnimation.StandardEasing
        }
        ColorAnimation {
            target: mainView
            property: "backgroundColor"
            from: backgroundColor
            to: newBackgroundColor
            duration:UbuntuAnimation.BriskDuration
            easing: UbuntuAnimation.StandardEasing
        }
        ColorAnimation {
            target: mainView
            property: "footerColor"
            from: footerColor
            to: newFooterColor
            duration:UbuntuAnimation.BriskDuration
            easing: UbuntuAnimation.StandardEasing
        }
    }

    function adjustBackground(degrees) {
        gradientAnimation.stop()
        var celsius = (settings.units === "imperial") ? Math.round((degrees-32)/1.8) : degrees,
            colors = Gradients.getGradients(celsius);
        newHeaderColor = colors.header;
        newBackgroundColor = colors.body;
        newFooterColor = colors.footer;
        gradientAnimation.start()
    }

    function responseDataHandler(messageObject) {
        if(!messageObject.error) {
            if(messageObject.action === "updateData") {
                messageObject.result.forEach(function(loc) {
                    // replace location data in cache with refreshed values
                    if(loc["save"] === true) {
                        storage.updateLocation(loc.db.id, loc);
                    }
                });
                buildTabs(messageObject.result);
            }
        } else {
            console.log(messageObject.error.msg+" / "+messageObject.error.request.url)
            httpFailedSplash.show()
            refreshData(true, null);
        }
    }

    // see https://bugs.launchpad.net/ubuntu-ui-toolkit/+bug/1124071, temporary workaround    
    function buildTabs(locations) {
        // check if there was a location added
        var locLength = locations.length,
            locBeforeLen = locationsList.length,
            focusToLast = (locBeforeLen > 0 && locLength > locBeforeLen) ? true : false;
        refresh.visible = false;

        // assure the right color is set again after reload, see refresh()
        Theme.palette.selected.backgroundText = "#656565";

        // show locationmanager when no location is added
        if(locLength === 0) {
            showLocationManager()
            return;
        }
        locationsList = locations;
        // reset the tab reapeater to get a full rebuild
        tabRepeater.model = 0;
        // add/re-add the tabs to the repeater
        tabRepeater.model = locLength;
        if(locLength > 0) {
            for(var x=0;x<locLength;x++) {
                var tab = tabRepeater.itemAt(x);
                tab.title = locations[x].location.name;
                tab.locationIndex = x;
                tab.ready();
            }
        }
        if(focusToLast) {
            tabs.selectedTabIndex = locLength -1;
        } else if(tabIndexAtRefresh > -1 && tabIndexAtRefresh < tabRepeater.count) {
            tabs.selectedTabIndex = tabIndexAtRefresh
            tabIndexAtRefresh = -1
        } else {
            tabs.selectedTabIndex = 0;
        }
    }


    function refreshData(from_storage, force_refresh) {
        refresh.visible = true;
        if(from_storage === true && force_refresh !== true) {
            storage.getLocations(buildTabs);
        } else {            
            storage.getLocations(function(locations) {
                // set this color temporary to white, to keep sandwich icon white until tabs are rebuild
                Theme.palette.selected.backgroundText = "#FFFFFF";
                WeatherApi.sendRequest({
                    action: "updateData",
                    params: {
                        locations:locations,
                        force:force_refresh,
                        service:settings["service"],
                        api_key: Key.twcKey
                    }
                }, responseDataHandler)
                //locationDataWorker.loadLocations(locations, force_refresh)
            });
        }
    }

    function showLocationManager() {
        tabIndexAtRefresh = tabs.selectedTabIndex
        pageStack.push(Qt.resolvedUrl("components/LocationManagerPage.qml"))
    }

    function showSettings() {
        tabIndexAtRefresh = tabs.selectedTabIndex
        pageStack.push(Qt.resolvedUrl("components/SettingsPage.qml"))
    }

    Component.onCompleted: {
        //storage.clearDB();
        //storage.clearSetting('units');

        Theme.palette.selected.backgroundText = "#656565";
        Theme.palette.normal.backgroundText = "#FFFFFF";
        Theme.palette.normal.background = "#ECECEC"

        storage.getSettings(function(storedSettings) {            
            for(var settingName in storedSettings) {
                settings[settingName] = storedSettings[settingName];
            }
            refreshData();
        })
    }
    
    Components.Storage{
        id: storage
    }    

    Components.SplashComponent {
        id:httpFailedSplash
        objectName: "HTTPFailedSplash"
    }

    Components.LoadingSpinnerComponent {
        id:refresh
    }

    PageStack {
        id:pageStack
        anchors.fill: parent
        Component.onCompleted: push(tabs)
        Tabs {
            id: tabs;
            objectName: 'rootTabs';

            Repeater {
                id: tabRepeater
                model: 0
                Components.LocationTab {}
            }
        }
    }

}
