import QtQuick 2.3
import QtQuick.XmlListModel 2.0

import Ubuntu.Components 1.1
import Ubuntu.Components.ListItems 1.0 as ListItems
import Ubuntu.Components.Popups 1.0

import "databasemodule_v2.js" as DB
import "./imgSeparator.js" as ImageUtils
import "./dateutils.js" as DateUtils

Item {
    id: networkManagerRoot

    signal downloadFinished(int tagId)
    signal downloadStarted(int tagId)

    property string operationStatus: "success"

    function updateFeeds(feedsArray, topicId) {
        d.updateFeeds(feedsArray, topicId)
    }

    /* All private method are inside QtObject.
     */
    QtObject {
        id: d

        property var feedList: []                 // Feed list to update.
        property var currentFeed                  // Current downloading feed.
        property int tagId: 0                     // Tag to update.

        /* Method updates feeds one by another.
         * Input: array of objects, each should include
         * source, link and id (of feed in DB) properties.
         */
        function updateFeeds(feedsArray, topicId) {
            tagId = topicId || 0

            downloadStarted(tagId)

            feedList = feedsArray
            operationStatus = "success"
            updateNextFeed()
        }

        // For inner usage only.
        function updateNextFeed() {
            if (feedList.length == 0) {
                downloadFinished(tagId)
                return
            }

            currentFeed = feedList.shift()
            googleFeedApi.loadFeed(currentFeed.source)
        }

        function cancelDownload() {
            feedList = []
            operationStatus = "abort"
            googleFeedApi.abort()
        }

        /* Temporary here - TODO move to separate file.
         */
        function updateFeedInfo(feedId, feedLink, responseData) {
            var entries = responseData.feed.entries
            var f = responseData.feed

            DB.updateFeedByXml(feedId, f.feedUrl === f.link ? feedLink : f.link, // Sometimes google fails and sends site link equal to feed url.
                                                              f.description, f.title)
            console.log(" -------- UPDATE INFO -------- ")
            console.log(f.title, f.link, f.feedUrl, f.description)

            // Save some needed properties before clear articles, currently only status.
            var articleProperties = []
            var articlePropertiesDb = DB.preloadArticlesProperties(feedId)
            for (var j = 0; j < articlePropertiesDb.rows.length; j++) {
                var itm = articlePropertiesDb.rows.item(j)
                articleProperties.push({ "guid" : itm.guid, "status" : itm.status })
            }

            DB.clearArticles(feedId)

            var articleModel = []
            for (var i = 0; i < entries.length; i++) {
                var e = entries[i]

                // Grab image from for article.
                var articleImage = grabArticleImage(e)
                e.content = clearFromBadTags(e.content)

                var temp =
                        {
                    "title": e.title,
                    "content": e.content,
                    "link": e.link,
                    "author": e.author,
                    "description": e.contentSnippet,
                    "pubDate": DateUtils.parseDate(e.publishedDate),
                    "guid": Qt.md5(e.content + e.publishedDate),
                    "image" : articleImage,
                    "media_groups" : e.mediaGroups //== undefined ? "" : JSON.stringify(e.mediaGroups),
                }

                articleModel.push(temp)
            }

            /* Add new articles to DB and restore 'read' status of some of them.
             */
            DB.addArticles(articleModel, feedId, articleProperties);
        }

        // Not naive anymore :)
        function grabArticleImage(e) {
            if (e.mediaGroups) {
                var medias = e.mediaGroups
                for (var i = 0; i < medias.length; i++) {
                    var media = medias[i]

                    for (var j = 0; j < media.contents.length; j++) {
                        var cont = media.contents[j]

                        if (cont.type === "image/jpeg" || cont.type === "image/png" ||
                                cont.type === "image/jpeg" || cont.type === "image/pjpeg" ||
                                cont.type === "image/svg+xml" || cont.medium === "image") {
                            return cont.url
                        }
                    }
                }
            }
            // console.log("getFirstImage")
            // console.log(e.content)
            var image = ImageUtils.getFirstImage(e.content)
            if (image != null) {
                return image
            }

            return null
        }

        function clearFromBadTags(content) {
            /* Remove non empty too. Useless anyway.
             */
            content = content.replace(/alt=".*?"/g, "")
            content = content.replace(/title=".*?"/g, "")
            return content
        }

        property var googleFeedApi: GoogleFeedApi {
            onLoadResult: {
                if (result.responseStatus !== 200) {
                    console.log("XML NETWORK GFA:", JSON.stringify(result))
                    if (operationStatus == "success")
                        operationStatus = "withErrors"
                } else d.updateFeedInfo(d.currentFeed.id, d.currentFeed.link, result.responseData)

                d.updateNextFeed()
            }
        } // GFA
    } // QtObject
}
