import QtQuick 2.3
import QtQuick.XmlListModel 2.0
import Ubuntu.Components 1.1
import Ubuntu.Components.ListItems 1.0 as ListItem
import Ubuntu.Components.Popups 1.0
import "./feeds"
import "./listview"

import "./databasemodule_v2.js" as DB
import "./imgSeparator.js" as ImageUtils

MainView {
    id: mainView

    property bool isTabletMode: false // Disabled for now.
//    property bool isTabletMode: {
//        if (width === 0 || height === 0)
//            return
//        return (width > units.gu(90))
//    }

    readonly property real latestDBVersion: 1.2

    objectName: "mainView"
    applicationName: "com.ubuntu.shorts"
    anchorToKeyboard: true

    width: units.gu(100)
    height: units.gu(80)
    focus: true

    useDeprecatedToolbar: false

    headerColor: "#3a2c32"
    backgroundColor: "#875864"
    footerColor: "#9b616c"

    Component.onCompleted: {

        // Update database if the db scheme is old.
        var dbv = parseFloat(optionsKeeper.dbVersion())
        console.log("db version: " , dbv)
        if (dbv < latestDBVersion) {
            if (DB.updateDatabase(dbv)) {
                optionsKeeper.setDbVersion(latestDBVersion)
            }
            else {
                // TODO if update database fail ??
                // TODO We can drop tables and create new
            }
        }

        // Filling DB with default data.
        //DB.dropTable()
        var result = DB.feedsCount()

        if (result.count < 1) {
            DB.addTag("Ubuntu")
            DB.addFeed("" , "http://developer.ubuntu.com/feed/")
            DB.addFeed("" , "http://design.canonical.com/feed/")
            DB.addFeedTag(1, 1)
            DB.addFeedTag(2, 1)
            DB.addTag("Canonical")
            DB.addFeed("" , "http://voices.canonical.com/feed/atom/")
            DB.addFeed("" , "http://insights.ubuntu.com/feed/")
            DB.addFeed("" , "http://blog.canonical.com/feed/")
            DB.addFeedTag(3, 2)
            DB.addFeedTag(4, 2)
            DB.addFeedTag(5, 2)
            topicManagement.reloadTopics()
            refresh()
        } else {
            repeater.reloadTabs()
            /* Update if "isListView" set will be done automatically.
             */
            if (!pageStack.isListView) {
                shortsTab.reload()
                savedTab.reload()
            }
        }
    }

    /* Refresh current topic or all feeds if we are in the Shorts.
     */
    function refresh() {
        var topicId = 0
        if (pageStack.currentPage == tabstabs) {
            var curTab = tabstabs.selectedTab
            topicId = curTab.topicId
        }

        console.log("REFRESH TOPIC", topicId)
        if (topicId == -2) {
            refreshSavedTab()
            return
        }

        var feeds
        if (topicId !== 0)
            feeds = DB.loadFeedsFromTag(topicId)
        else feeds = DB.loadFeeds()

        var feedarray = []
        for (var i=0; i< feeds.rows.length; i++) {
            feedarray.push(feeds.rows.item(i))
        }
        networkManager.updateFeeds(feedarray, topicId)
    }

    function reloadViews(tagId) {
        // savedTab.reload() // Doesn't change with network update.
        tagId = tagId || 0

        if (tagId == 0 || !repeater.containsTopic(tagId)) {
            shortsTab.reload()
            repeater.reloadTabs()
        } else {
            shortsTab.reload()
            repeater.reloadOneTopic(tagId)
        }

        tabstabs.selectTopic(tagId)
    }

    // refresh "Saved" Tab
    function refreshSavedTab() {
        savedTab.reload()
    }

    function showArticle(model, index) {      //   go to single article page
        articlePage.setFeed(model, index)
        pageStack.push(articlePage)
    }

    function editFeed(feedid, title, url, pTopicId) {
        editFeed.setValues(feedid, title, url, pTopicId) ;
        pageStack.push(editFeed) ;
    }

    //  change the topic name if the name changed by management
    function changeTopicName(topicId, newName) {
        repeater.changeTopicName(topicId, newName)
    }

    /* -------------------------- Visuals ---------------------------- */

    PageStack {
        id: pageStack
        objectName: "pageStack"

        property bool isListView: optionsKeeper.useListMode()
        property var commonHeadActions: [refreshAction, changeModeAction, addReadsAction, editTopicsAction]

        anchors.fill: parent
        focus: true

        Component.onCompleted: {
            push(tabstabs)
        }

        Keys.onPressed: {
            console.log(objectName, event.key.toString(16), event.modifiers.toString(16))

            /* Hotkeys for tabs management.
             */
            if (currentPage == tabstabs) {
                if (event.key >= Qt.Key_0 && event.key <= Qt.Key_9 &&
                        event.modifiers & Qt.AltModifier) { // Digits for switching tabs...
                    var tabNumber = event.key - Qt.Key_0
                    if (tabNumber < tabstabs.getTabCount()) {
                        tabstabs.selectedTabIndex = tabNumber
                        event.accepted = true
                    }
                }

                if (event.modifiers & Qt.ControlModifier) { // Control+Tab and Control+Shift+Tab
                    if (event.key === Qt.Key_Backtab) {
                        tabstabs.selectedTabIndex = (tabstabs.selectedTabIndex - 1 + tabstabs.getTabCount()) % tabstabs.getTabCount()
                        event.accepted = true
                    } else if (event.key === Qt.Key_Tab) {
                        tabstabs.selectedTabIndex = (tabstabs.selectedTabIndex + 1) % tabstabs.getTabCount()
                        event.accepted = true
                    } else if (event.key === Qt.Key_N) {
                        pageStack.push(appendFeedPage, {"isDirty" : true})
                    } else if (event.key === Qt.Key_R) {
                        refresh()
                    }
                }
            } else if (currentPage == articlePage) {
                if (event.key == Qt.Key_Left) {
                    articlePage.articleView.showPrevArticle()
                } else if (event.key == Qt.Key_Right) {
                    articlePage.articleView.showNextArticle()
                }
            }
        }

        onCurrentPageChanged: {
            if (currentPage != null) {
                if (currentPage.reloadPageContent)
                    currentPage.reloadPageContent()
            }
        }

        /* -------------------------- Actions ---------------------------- */

        Action {
            id: refreshAction

            text:  i18n.tr("Refresh")
            iconSource: Qt.resolvedUrl("./icons_tmp/reload.svg")
            onTriggered: refresh()
        }

        Action {
            id: changeModeAction
            text:  pageStack.isListView ? i18n.tr("Grid View") : i18n.tr("List view")
            iconSource: pageStack.isListView ? Qt.resolvedUrl("./icons_tmp/view-fullscreen.svg") : Qt.resolvedUrl("./icons_tmp/view-fullscreen.svg")
            onTriggered: {
                pageStack.isListView = !pageStack.isListView
                optionsKeeper.setUseListMode(pageStack.isListView)


                /* Spike solution for bug: https://bugs.launchpad.net/ubuntu-rssreader-app/+bug/1214916
                                 * If we switched to grid, manually cause header return.
                                 */
                if (!pageStack.isListView) {
                    var ind = tabstabs.selectedTabIndex
                    if (ind == 0) {
                        tabstabs.selectedTabIndex = 1
                        tabstabs.selectedTabIndex = ind
                    } else {
                        tabstabs.selectedTabIndex = 0
                        tabstabs.selectedTabIndex = ind
                    }
                } else if (tabstabs.selectedTabIndex > 1) {
                    /* If we swithed to list, there are no selectedTabIndexChanged signal
                                     * will occur. So we must ensure that content of selected tab is displayed
                                     * (in TopicTab only, Saved and Shorts already have visuals).
                                     */
                    repeater.itemAt(tabstabs.selectedTabIndex - 2).showContent()
                }

                // mainCommonToolbar.opened = false
            } // onTriggered
        }

        Action {
            id: addReadsAction
            iconSource: Qt.resolvedUrl("./icons_tmp/add.svg")
            text: i18n.tr("Add feeds")
            onTriggered: {
                /* When we want to show "Append feed" page from
                 * any page except of "Choose topic",
                 * we should clear it (mark as dirty).
                 * ATTENTION: similar call can be found in hotkeys section.
                 */
                pageStack.push(appendFeedPage, {"isDirty" : true})
            }
        }

        Action {
            id: editTopicsAction
            objectName:"editTopicsAction"
            text:  i18n.tr("Edit topics")
            iconSource: Qt.resolvedUrl("./icons_tmp/edit.svg")
            onTriggered: {
                pageStack.push(topicManagement)
            }
        }

        /* -------------------------- Pages & Tabs ---------------------------- */

        Tabs {
            id: tabstabs

            objectName: "tabstabs"
            visible: false

            Component.onCompleted: {
                selectedTabIndex = 1
            }

            onSelectedTabIndexChanged: {
                // console.log("onSelectedTabIndexChanged", selectedTabIndex)
                if (selectedTabIndex <= 1)
                    return
                repeater.itemAt(selectedTabIndex - 2).showContent()
            }

            function getTabCount() {
                return repeater.model + 2 // + Shorts and Saved
            }

            function selectTopic(topicId) {
                if (/*topicId == undefined ||*/ topicId == 0 ) {
                    tabstabs.selectedTabIndex = 1
                } else {
                    for(var i = 0; i < repeater.count; i++) {
                        if (repeater.itemAt(i).topicId == topicId) {
                            tabstabs.selectedTabIndex = i + 2
                            repeater.itemAt(i).showContent()
                            break
                        }
                    }
                }
            }

            /* Used for keeping status in sync across tabs.
             */
            function updateStatusInModels(tagId, articleId, status) {
                savedTab.updateStatusInModel(articleId, status)
                shortsTab.updateStatusInModel(articleId, status)
                for(var i = 0; i < repeater.count; i++) {
                    if (repeater.itemAt(i).topicId == tagId) {
                        repeater.itemAt(i).updateStatusInModel(articleId, status)
                        break
                    }
                }
            }

            /* Used for keeping favourite in sync across tabs.
             */
            function updateFavouriteInModels(article, fav) {
                var tagId = article.tagId
                var articleId = article.id

                savedTab.updateFavouriteInModel(article, fav)
                shortsTab.updateFavouriteInModel(articleId, fav)
                for(var i = 0; i < repeater.count; i++) {
                    if (repeater.itemAt(i).topicId == tagId) {
                        repeater.itemAt(i).updateFavouriteInModel(articleId, fav)
                        return
                    }
                }
            }

            SavedTab {
                id: savedTab

                objectName: "Tab1"
                title: i18n.tr("Saved")
                topicId: -2
            }

            ShortsTab {
                id: shortsTab

                objectName: "Tab0"
                title: i18n.tr("Shorts")
                topicId: 0
            }

            Repeater {
                id: repeater

                TopicTab { }

                function reloadTabs() {
                    repeater.model = 0
                    var tags = DB.loadTags()
                    repeater.model = tags.rows.length

                    for (var i = 0; i< tags.rows.length; i++) {
                        repeater.itemAt(i).title = tags.rows.item(i).name
                        repeater.itemAt(i).topicId = tags.rows.item(i).id
                        repeater.itemAt(i).reload()
                    }
                }

                function changeTopicName(topicId, newName) {
                    for(var i = 0; i < repeater.count; i++) {
                        if (repeater.itemAt(i).topicId == topicId) {
                            repeater.itemAt(i).title = newName
                            break
                        }
                    }
                }

                function reloadOneTopic(topicId) {
                    for(var i = 0; i < repeater.count; i++) {
                        if (repeater.itemAt(i).topicId == topicId) {
                            repeater.itemAt(i).reload()
                            break
                        }
                    }
                }

                function containsTopic(topicId) {
                    for(var i = 0; i < repeater.count; i++) {
                        if (repeater.itemAt(i).topicId == topicId) {
                            return true
                        }
                    }

                    return false
                }
            } // Repeater
        } // Tabs

        // ******************************** Create Topic **********************///////////////

        CreateTopicPage {
            id: createTopicPage
            visible: false
            flickable: null
        } // Page

        // ******************************** APPEND FEED ***********************///////////////

        AppendFeedPage {
            id: appendFeedPage

            title: i18n.tr("Add feeds")
            flickable: null
            visible: false
        }

        // ******************************** Choose Topic Page ***********************///////////////

        ChooseTopicPage {
            id: chooseTopicPage
            visible: false
        }

        // ******************************** Rss Feed Page ***********************///////////////

        ArticleViewPage {
            id: articlePage
            visible: false

            Connections {
                target: articlePage.articleView

                onArticleStatusChanged: {
                    tabstabs.updateStatusInModels(tagId, articleId, status)
                }

                onArticleFavouriteChanged: {
                    tabstabs.updateFavouriteInModels(article, favourite)
                }
            }
        }

        // ******************************** Topic Management Page ***********************///////////////
        TopicManagement {
            id: topicManagement
            visible: false
        }

        // ******************************** Edit Feed Page ***********************///////////////
        EditFeedPage {
            id: editFeed
            visible: false
        }
    } // PageStack

    /* -------------------------- Utils ---------------------------- */

    NetworkManager {
        id: networkManager

        onDownloadFinished: {
            reloadViews(tagId)
        }

        onDownloadStarted: {
            PopupUtils.open(refreshWaitDialogComponent, null)
        }
    }

    OptionsKeeper {
        id: optionsKeeper
    }

    /* -------------------------- Components ---------------------------- */

    Component {
        id: refreshWaitDialogComponent

        Dialog {
            id: refreshWaitDialog
            objectName: "refreshWaitDialog"

            title: i18n.tr("Checking for new articles")

            ActivityIndicator {
                objectName: "activityindicator"
                running: true
            }

            Button {
                text: i18n.tr("Cancel")
                objectName: "refreshCancel"
                color: UbuntuColors.orange
                onClicked: {
                    networkManager.cancelDownload()
                    PopupUtils.close(refreshWaitDialog)
                }
            }

            Connections {
                target: networkManager

                onDownloadFinished: {
                    PopupUtils.close(refreshWaitDialog)
                    if ( networkManager.operationStatus === "withErrors" ) {
                        PopupUtils.open(errorDialogComponent, appendFeedPage,
                                        {"text" : i18n.tr("Perhaps some of the channels have not been updated."),
                                            "title" : i18n.tr("Errors occurred during the update")})
                    }
                }
            }
        } // Dialog
    } // Component

    Component {
        id: addReadsPopoverComp

        ActionSelectionPopover {
            id: addReadsPopover
            objectName: "addreadspopover"

            actions: ActionList {
                id: popoverActionsList

                Action {
                    text: i18n.tr("+ Add feeds")
                    onTriggered: {
                        /* When we want to show "Append feed" page from
                         * any page except of "Choose topic",
                         * we should clear it (mark as dirty).
                         * ATTENTION: similar call can be found in hotkeys section.
                         */
                        pageStack.push(appendFeedPage, {"isDirty" : true})
                    }
                }

                // Currently redundant. We will use it later.
                //                Action {
                //                    text: i18n.tr("+ New topic")
                //                    onTriggered: {
                //                        pageStack.push(createTopicPage)
                //                    }
                //                }

                Action {
                    text: i18n.tr("Add online accounts")
                    onTriggered: {
                        PopupUtils.open(errorDialogComponent, null,
                                        {"text" : i18n.tr("Online accounts are not available for now"),
                                            "title" : i18n.tr("We are sorry")})
                    }
                }

                Action {
                    text: i18n.tr("Import subscriptions")
                    onTriggered: {
                        PopupUtils.open(errorDialogComponent, null,
                                        {"text" : i18n.tr("Importing subscriptions is not available for now"),
                                            "title" : i18n.tr("We are sorry")})
                    }
                }
            } // ActionList
        } // ActionSelectionPopover
    } // Component

    Component {
        id: errorDialogComponent

        Dialog {
            id: errorDialog

            Button {
                text: i18n.tr("Ok")
                objectName: "errordialogbutton"
                onClicked: PopupUtils.close(errorDialog)
            }
        }
    } // Component

    /* -------------------------- Connections ---------------------------- */

    Connections {
        target: createTopicPage

        onTopicAdded: {
            reloadViews()
        }
    }

    Connections {
        target: chooseTopicPage

        onTopicChoosen: {
            networkManager.updateFeeds(addedFeeds, topicId)
        }
    }

    Connections {
        target: topicManagement

        onFeedEdit: {
            console.log("onFeedEdit: ", topicId)
            shortsTab.reload()
            repeater.reloadOneTopic(topicId)
            tabstabs.selectTopic(topicId)
        }

        onTopicDeleted: {
            reloadViews()
        }
    }

    // Workaround for https://bugs.launchpad.net/ubuntu-ui-toolkit/+bug/1201400.
    // Inspired by the KeyboardRectangle component, originally written for the
    // telephony application.
    Connections {
        target: Qt.inputMethod

        onVisibleChanged: {
            //            console.log("keyboard visible: ", Qt.inputMethod.keyboardRectangle.height)
            if (!Qt.inputMethod.visible) {
                var focusedItem = recursiveFindFocusedItem(window);
                if (focusedItem != null) {
                    focusedItem.focus = false;
                }
            }
        }

        function recursiveFindFocusedItem(parent) {
            if (parent.activeFocus) {
                return parent;
            }

            for (var i in parent.children) {
                var child = parent.children[i];
                if (child.activeFocus) {
                    return child;
                }

                var item = recursiveFindFocusedItem(child);

                if (item != null) {
                    return item;
                }
            }

            return null;
        }
    }
}
